/**
 *#########################################################################
 * FedoraToGS3Interface.java - works with the demo-client for Greenstone 3, 
 * of the Greenstone digital library suite from the New Zealand Digital 
 * Library Project at the  * University of Waikato, New Zealand.
 * <BR><BR>
 * Copyright (C) 2008 New Zealand Digital Library Project
 * <BR><BR>
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * <BR><BR>
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *########################################################################
 */

package org.greenstone.fedora.services;

import java.util.Map;

/**
 * Most of the following methods return the same data as FedoraGS3DL, but 
 * formatted as Greenstone Response-Message XML. This way our java-client can 
 * parse the returned XML in the same way and store them in the same data
 * structures. 
 * @author ak19
*/
public interface FedoraToGS3Interface // extends FedoraGS3DL 
{
	/** Interface declaring some string constants of literals to look for */
	public interface Constants {
		public static final String GS3FilePathMacro = "_httpdocimg_";
		/** Prefix of the (img) files associated  with Greenstone documents 
		 * in the Fedora repository */
		public static final String ASSOCFILEPREFIX = "FG";
		
		/** names of query form controls */
		public static final String MAXDOCS = "maxDocs";
		public static final String QUERY = "query";
		public static final String SIMPLEFIELD_ATT = "simpleField";
		public static final String OCCURS_ATT = "occurs";
		public static final String FIELDNAME_ATT = "fieldName";
		public static final String NUM_DOCS_MATCHED = "numDocsMatched";
		
		/** Field search names and separator used externally */
		public static final String ALL_FIELDS = "allFields";
		public static final String DOC_TITLES = "docTitles"; // doc titles only
		public static final String ALL_TITLES = "allTitles"; // all titles, section & doc
		public static final String FULLTEXT = "fullText";
		public static final String COMMA = ",";
	}
    
        /* FEDORA GENERIC SEARCH RELATED */
	/** @return the gSearchWSDLURL, the url of the WSDL for the
	 * FedoraGSearch web services */
	public String getGSearchWSDLURL();
	
	/** Sets the member variable gSearchWSDLURL that specify the location of
	 * the WSDL file of FedoraGSearch's web services. Then it attempts 
	 * to instantiate a connection to those web services. 
	 * @param url is the new url of the GSearch web services WSDL file */
	public void setGSearchWSDLURL(String url);
	
	/** @return the gSearchIndexName, the name of the index Fedora Generic
	 * Search will search in (where GS3 docs have been indexed into). */
	public String getGSearchIndexName();
	
	/** Sets the member variable gSearchIndexName that specifies the name
	 * of the index containing indexed GS3 documents. Then it attempts 
	 * to instantiate a connection to the Fedora GSearch web services using
	 * this changed value for indexName. 
	 * @param indexName is the new name of the index containing indexed GS3 
	 * docs that GSearch should search in. */
	public void setGSearchIndexName(String indexName);
	
	/* DESCRIBE SERVICES */
	/** @return a GS3 response message for a describe services request: 
	 * indicating the list of services supported by the Fedora-Greenstone 
	 * interface. These are DocumentContentRetrieve, DocumentMetadataRetrieve,
	 * DocumentStructureRetrieve, BrowseByLetter - as indicated by member
	 * variable serviceNames. */
	public String getServiceList();

	/** @return a GS3 describe response message listing the collections and  
	 * collection-specific metadata stored in the Fedora-Greenstone repository. */
	public String getCollectionList();

	/** @return a GS3 describe response message for a collection in the 
	 * Fedora-Greenstone repository. 
	 * @param collectionName - the name of the collection that is to be described.  
	 * It will be converted to a fedora collection pid, which is of the form 
	 * "greenstone:&lt;collectionName&gt;-collection". */
	public String describeCollection(String collectionName);
	
	/** @return a GS3 describe response message for the services of a collection 
	 * in the Fedora-Greenstone repository. So far, these services are the same for
	 * all fedora collections: they are the services given in member variable
	 * serviceNames: DocumentContent/Metadata/StructureRetrieve, browse.   
	 * @param collectionName - the name of the collection whose services are to
	 * be described. It will be converted to a fedora collection pid, which is of 
	 * the form "greenstone:&lt;collectionName&gt;-collection". */
	public String describeCollectionServices(String collectionName);

	/** 
	 * All collections in this Digital Library (Fedora Repository) share 
	 * the same services, so this method returns the same as 
	 * describeService(serviceName). 
	 * @return a GS3 describe response message for the requested service 
	 * of the given collection. DocumentContent/Metadata/StructureRetrieve 
	 * return nothing special except their names; browse (and any query)
	 * return more complex XML responses.   
	 * @param collectionName - the name of the collection whose service is to
	 * be described. It will be converted to a fedora collection pid, which is of 
	 * the form "greenstone:&lt;collectionName&gt;-collection". 
	 * @param serviceName - the name of the service in the collection which is to 
	 * be described. */
	public String describeCollectionService(String collectionName, 
			String serviceName);
	
	/** All collections in this Digital Library (Fedora Repository) share 
	 * the same services, so this method returns the same as 
	 * describeCollectionService(collName, serviceName). 
	 * @return a GS3 describe response message for the requested service 
	 * of the given collection. DocumentContent/Metadata/StructureRetrieve 
	 * return nothing special except their names; browse (and any query)
	 * return more complex XML responses.
	 * All collections in this Digital Library (Fedora Repository) share 
	 * the same services, so this method returns the same as 
	 * describeService(serviceName). 
	 * @param serviceName - the name of the service in the collection which is to 
	 * be described.*/
	public String describeService(String serviceName);
	
	/* RETRIEVE SERVICE: DOCUMENT CONTENT RETRIEVE */
	
	/** Given a list of document identifiers that are either docPIDs or  
	 * concatenations of docPID+sectionID, this method retrieves their contents.
	 * @param docID is expected to be of the form 
	 * "greenstone:&lt;collectionName&gt;-&lt;docPID&gt;-&lt;sectionNumber&gt;" or
	 * "greenstone:&lt;collectionName&gt;-&lt;docPID&gt;"
	 * If it is "greenstone:&lt;collectionName&gt;-&lt;docPID&gt;", then the content for
	 * "greenstone:&lt;collectionName&gt;-1" ("greenstone:&lt;collectionName&gt;-Section1")
	 * is returned. */
	public String getContent(String docID);
	
	/** Given a document identifier that is either a docPID or a concatenation  
	 * of docPID+sectionID, this method retrieves the content for it.
	 * @param docIDs an array of document IDs where each is expected to be of the  
	 * form "greenstone:&lt;collectionName&gt;-&lt;docPID&gt;-&lt;sectionNumber&gt;" 
	 * or "greenstone:&lt;collectionName&gt;-&lt;docPID&gt;"
	 * If it is "greenstone:&lt;collectionName&gt;-&lt;docPID&gt;", then the content for
	 * "greenstone:&lt;collectionName&gt;-1" ("greenstone:&lt;collectionName&gt;-Section1")
	 * is returned. */
	public String getContent(String[] docIDs);
	
	/* RETRIEVE SERVICE: DOCUMENT STRUCTURE RETRIEVE */

    	/** @return a String representing Greenstone3 DocumentMetadataRetrieve XML
	 * containing the requested portion of the document structure of the documents 
	 * indicated by docIDs: 
	 * @param collection is the name of the collection. It's already included in the
	 * docID for a Fedora DL.
	 * @param docID is the document identifier of the document whose hierarchical
	 * structure is requested. The collection name is already included in the
	 * docID for a Fedora DL.
	 * @param structure - strings specifying the required structure of the document. 
	 * Any combination of: ancestors, parent, siblings, children, descendants, entire.
	 * @param info - strings specifying the required structural info of the document. 
	 * It can be any combination of: siblingPosition, numSiblings, numChildren.
	*/
	public String getDocumentStructure(String docID, String[] structure, String[] info);

	/** @return a String representing Greenstone3 DocumentMetadataRetrieve XML
	 * containing the requested portion of the document structure of the documents 
	 * indicated by docIDs:
	 * @param docIDs is an array of document identifiers of documents whose
	 * hierarchical structures are requested. The collection name is already included
	 * in the docID for a Fedora DL. 
	 * @param structure - strings specifying the required structure of each document. 
	 * Any combination of: ancestors, parent, siblings, children, descendants, entire.
	 * @param info - strings specifying the required structural info of each document. 
	 * It can be any combination of: siblingPosition, numSiblings, numChildren.
	*/
	public String getDocumentStructure(String[] docIDs, String[] structure, String[] info);

	
	/* RETRIEVE SERVICE: DOCUMENT METADATA RETRIEVE */
	
	/** Given a list of collectionIDs, returns a GS3 DocumentMetadataRetrieve 
	 * response message that gives the metadata for each collection identified
	 * @param collIDs is an array of fedora pids identifying collections in the
	 * fedora repository 
	 * @return a GS3 DocumentMetadataRetrieve response message containing the
	 * EX metadata for all the requested collections */
	public String getCollectionMetadata(String[] collIDs);
	
	/** Given a list of document identifiers, a GS3 DocumentMetadataRetrieve 
	 * response message is returned containing the metadata for each document. 
	 * @param docIDs is an array of document identifiers (docID can either be &lt;pid&gt;s 
	 * items (documents) in the fedora repository, or "&lt;pid&gt;-sectionNumber". 
	 * @return a GS3 DocumentMetadataRetrieve response message containing the
	 * EX, DC, DLS metadata for all the requested documents */
    public String getDocumentMetadata(String[] docIDs, String[] metadata);
	
	/** Given a collectionID, returns a GS3 DocumentMetadataRetrieve 
	 * response message that gives the metadata for the collection identified
	 * @param collID is a fedora pid identifying a collection in its repository
	 * @return a GS3 DocumentMetadataRetrieve response message containing the
	 * EX metadata for the requested collection */
	public String getCollectionMetadata(String collID);
	
	/** Given a document identifier, returns a GS3 DocumentMetadataRetrieve 
	 * response message containing the metadata for the document. 
	 * @param docID is a document identifier (docID can either be a &lt;pid&gt; 
	 * of an item (document) in the fedora repository, or it can be 
	 * "&lt;pid&gt;-sectionNumber". 
	 * @return a GS3 DocumentMetadataRetrieve response message containing the
	 * EX, DC, DLS metadata for all the requested document */
    public String getDocumentMetadata(String docID, String[] metadata);

	/** @return a greenstone DocumentMetadataRetrieve response for the 
	 * documents or collections indicated by the docIDsOrCollIDs.
	 * @param docIDsOrCollIDs is an array of identifiers which may be either the
	 * fedora pids for collections, or otherwise may be a document identifier.
	 * In the last case, the document ID may consist of either
	 * "documentPID-sectionNumber" or may just be just fedora documentPID */
    public String getMetadata(String[] docIDsOrCollIDs, String[] metadata);
	
	/** Given a document identifier, returns a GS3 DocumentMetadataRetrieve 
	 * response message containing ONLY the Title metadata for the document. 
	 * @param docID is a document identifier (docID can either be a &lt;pid&gt;
	 * of an item (document) in the fedora repository, or it can be 
	 * "&lt;pid&gt;-sectionNumber". 
	 * @return a GS3 DocumentMetadataRetrieve response message containing the
	 * Title metadata for the requested document */
	public String getTitleMetadata(String docID);
	
	/** Given a document identifier, returns a GS3 DocumentMetadataRetrieve 
	 * response message containing ONLY the Title metadata for the documents. 
	 * @param docIDs is a list of document identifiers (where docID can either be 
	 * a &lt;pid&gt; of an item (document) in the fedora repository, or it can be 
	 * "&lt;pid&gt;-sectionNumber". 
	 * @return a GS3 DocumentMetadataRetrieve response message containing the
	 * Title metadata for all the requested documents */
	public String getTitleMetadata(String[] docIDs);
		
	/* BROWSE AND QUERY SERVICES */
	
	/** This method performs something equivalent to a greenstone3 
	 * ClassifierBrowseMetadataRetrieve on the classifierNodeIDs
	 * @param classNodeIDs are the IDs of the classifierNode for which the metadata
	 * needs to be returned
	 * @param metafields are the classifier metadata fields that are to be returned.
	 * @return a GS3 ClassifierBrowseMetadataRetrieve response message which 
	 * lists the metadata for all the classifierNodes passed as parameter.*/
	public String browseMetadataRetrieve(String[] classNodeIDs, String[] metafields);
		
	/** This method performs the implemented browse operation: allowing the
	 * user to browse the titles of documents in the given collection by letter
	 * and returning the results.
	 * @param classifierIDs are the ids of the classifiers on which to browse. In
	 * this case, the classifier indicates whether we browse titles by letter, or
	 * browse (documents) by collection; and it is of the form &lt;CL(letter)&gt;.
	 * @param collectionName is the name of the collection whose documents 
	 * starting with the given letter will be returned.
	 * @param structure - the requested browse substructure. Can be any combination
	 * of ancestors, parent, siblings, children, descendants.
	 * @param info - the requested structural info. Can be numSiblings, siblingPosition,
	 * numChildren
	 * @return a GS3 DocumentStructureRetrieve response message which lists all 
	 * the documents that start with the letter indicated by parameter classifier.
	*/
	public String browse(String collectionName, String[] classifierIDs, String[] structures, String[] infos);
	
	/* @returns the document identifiers returned for a search on the titles of 
	 * documents that contain the given term in their title */
	//public String queryTitle(String termInTitle) throws Exception;
	
	/**	@return a String representing Greenstone3 XML for a query process
	 * response returning the results for the query denoted by parameter
	 * nameValParamsMap.
	 * @param nameValParamsMap is a Map of name and value pairs for all the
	 * query field data values. The names match the field names that
	 * describeCollectionService() would have returned for the query service. 
	 * @param collection is the name of the collection 
	 * @param service is the name of the query service
	 * This method is only ever called when any of the services in the digital 
	 * library described themselves as type=query. Therefore any digital
	 * libraries that have no query services, can just return emtpy message 
	 * strings (or even "") since this method will never be called on them 
	 * anyway. */
	public String query(String collection, String service, 
			Map nameValParamsMap); 

    
    /** Given a URL that represents a fedoraPID, will look up the object.
     * If it exists, it will return the contents of the DC:Title of its datastream.
     * If it doesn't exist, it will return the URL as-is.
     * @param URL: the URL that (after modification) represents a fedoraPID to look up.
     * @param collection: the name of collection in which to search for the URL
     * representing a fedoraPID.
     * @return the string (representing a fedoraPID) stored in the DC:Title of the
     * URL-fedoraPID. If the URL-fedoraPID is not an object in the given collection,
     * then the parameter URL is returned.
    */
    public String getPIDforURL(String url, String collection);
}
