/*
 *    GSParams.java
 *    Copyright (C) 2008 New Zealand Digital Library, http://www.nzdl.org
 *
 *    This program is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
package org.greenstone.gsdl3.util;

import java.util.HashMap;
import java.util.ArrayList;

import org.apache.log4j.Logger;

/** keeps track of the servlet parameters, and their defaults */
public class GSParams
{

  static Logger logger = Logger.getLogger(org.greenstone.gsdl3.util.GSParams.class.getName());

  // Only used internally in the session
  public static final String USER_SESSION_CACHE = "user_session_cache";
  
	// cgi parameter names
	public static final String ACTION = "a"; // the major type of action- eg query or browse or process
	public static final String SUBACTION = "sa"; // subtype of action if we want different processing than the default
	public static final String REQUEST_TYPE = "rt"; // whether the request is just to display the service form, or to actually do a request to the service
	public static final String RESPONSE_ONLY = "ro"; // if == 1 do the request and pass back the response xml - no page formatting
	public static final String OUTPUT = "o"; // if processing is to be done, what type of output - html/xml/other??
	public static final String SERVICE = "s"; // the name of the service

  public static final String CACHE_KEY = "ck"; // if we want to use another cache key apart from the collection name
	public static final String UN = "un"; // username for authenticated-ping
	public static final String PW = "pw"; // pwd for authenticated-ping

	public static final String CLUSTER = "c"; // these two are the same
	public static final String COLLECTION = "c";
	public static final String COLLECTION_TYPE = "ct"; // collection type - mg, mgpp, lucene etc
  public static final String GROUP = "group";
	public static final String LANGUAGE = "l";
	public static final String DOCUMENT = "d";
	public static final String DOCUMENT_TYPE = "dt";
	public static final String HREF = "href"; // url. might be an external url, or a relative one that needs translating
	public static final String RELATIVE_LINK = "rl"; // whether the href url is relative to the collection or not.
	public static final String EXTERNAL_LINK_TYPE = "el"; // for an external link, go direct to the page or frame it in the collection
	public static final String PROCESS_ID = "pid"; // if a request wasn't completed, this identifies the request - used when asking for a status update

	public static final String HTTP_HEADER_FIELDS = "hhf";

  public static final String FAVOURITEBASKET = "favouritebasket";
  public static final String DOCUMENTBASKET = "documentbasket";
  public static final String WEBGLI_LINK = "webglilink";
    
	// internal configure args
	public static final String SYSTEM_SUBSET = "ss";
	public static final String SYSTEM_CLUSTER = "sc";
	public static final String SYSTEM_MODULE_NAME = "sn";
	public static final String SYSTEM_MODULE_TYPE = "st";

	// used for filtering out a piece of the final page
	public static final String EXCERPT_ID      = "excerptid";
	public static final String EXCERPT_ID_TEXT = "excerptid-text";
	public static final String EXCERPT_TAG     = "excerpttag";
	public static final String INLINE_TEMPLATE = "ilt";
	public static final String FILE_LOCATION = "fl";
  
	//Administration
	public static final String PASSWORD = "password";
	public static final String USERNAME = "username";
	public static final String GOOGLE_SIGNIN = "googleidentity-signin";
	public static final String LOGOUT = "logout";
	public static final String VERIFIED = "hmvf";

	// some standard arg values
	public static final String SYSTEM_ACTION = "s";

	// rss feeds
	public static final String RSS_ACTION = "rss";

	public static final String EXTERNAL_LINK_TYPE_DIRECT = "direct";
	public static final String EXTERNAL_LINK_TYPE_FRAMED = "frame";
	
	public static final String DEBUG = "debug";

  public static final String SERVICE_PREFIX   = "s1";
  public static final String PREVIOUS_PREFIX  = "p";
  public static final String MD_PREFIX        = "md___";
  public static final String SAVEPARAM_PREFIX = "jsession";

  public static final String SERVICE_PARAM_REGEX = "^s\\d\\..*";
  
  protected HashMap<String, Param> param_map = null;
  protected HashMap<String, Param> service_param_map = null;

    protected ArrayList<String> params_with_default_list = null;

	public GSParams()
	{
		this.param_map = new HashMap<String, Param>(30);
		this.service_param_map =  new HashMap<String, Param>(30);
		this.params_with_default_list = new ArrayList<String>(10);

		// we now only need to add in the ones that need saving, as we will default to "not save"
		// these ones are global params - apply across all collections
		//addParameter(name, save, sensitive, global)
		addParameter(LANGUAGE, true, false, true);
		addParameter(FAVOURITEBASKET, true, false, true);
                addParameter(DOCUMENTBASKET, true, false, true);
		addParameter(DEBUG, true, false, true);
		addParameter(WEBGLI_LINK, true, false, true); // ideally want set at servlet level

		// these will only be saved per collection/service
		addParameter(DOCUMENT, true);
		addParameter(PROCESS_ID, true);
		addParameter(COLLECTION_TYPE, true);
                addParameter(GROUP, true);
                
		// password is sensitive. don't save, but also don't return it in the page response
		addParameter(PASSWORD, false, true);
		
	}

	public boolean addParameter(String name, boolean save)
	{
	  return addParameter(name, "", save);
	}

        public boolean addParameter(String name, boolean save, boolean sensitive)
        {
		if (this.param_map.containsKey(name))
		{
			// already there so could not add
			return false;
		}
		this.param_map.put(name, new Param("", save, sensitive));
		return true;
	}
    
	public boolean addParameter(String name, String default_value, boolean save)
	{
		if (this.param_map.containsKey(name))
		{
			// already there so could not add
			return false;
		}

		this.param_map.put(name, new Param(default_value, save));
		if (default_value != null && !default_value.equals("")) {
		    this.params_with_default_list.add(name);
		}
		return true;
	}
  public boolean addParameter(String name, boolean save, boolean sensitive, boolean global)
  {
    if (this.param_map.containsKey(name))
      {
	// already there so could not add
	return false;
      }
    this.param_map.put(name, new Param("", save, sensitive, global));
    return true;
     
  }

  public boolean addServiceParameter(String name, String default_value, boolean save, boolean sensitive) {
    if (this.service_param_map.containsKey(name)) {
      // already there, could not add
      return false;
    }
    this.service_param_map.put(name, new Param(default_value, save, sensitive));
    return true;
  }
	
   
	public boolean setParamDefault(String name, String default_value)
	{
		Param p = this.param_map.get(name);
		if (p == null) {
		    addParameter(name, default_value, true);
		}else {
		    p.default_value = default_value;
		}
		if (!this.params_with_default_list.contains(name)) {
		    this.params_with_default_list.add(name);
		}
		return true;
	}

    public String getParamDefault(String name) {
	Param p = this.param_map.get(name);
	if (p==null) {
	    return null;
	}
	return p.default_value;
    
    }
	public boolean shouldSave(String name)
	{
		// p. is used to store previous settings
	        if (name.startsWith(PREVIOUS_PREFIX+".") || name.startsWith(MD_PREFIX)) {
			return false;
		}
		else if (name.startsWith(SAVEPARAM_PREFIX+".")) {
		    return true;
		}
		
		Param p;
		if (name.matches(SERVICE_PARAM_REGEX)) {
		  // its a service param
		  p = this.service_param_map.get(name.substring(3));
		} else {
		  // an ordinary param
		  p = this.param_map.get(name);
		}
		if (p==null) {
		  return false; // never save unknown params
		}
		
		return p.save;
	}
		

  public boolean isSensitive(String name) {
    Param p;
    if (name.matches(SERVICE_PARAM_REGEX)) {
      p = this.service_param_map.get(name.substring(3));
    } else {
      p = this.param_map.get(name);
    }
    if (p==null) {
      return false;
    }
    return p.sensitive;
  }

  public boolean isGlobal(String name) {
    Param p = this.param_map.get(name);
    if (p==null) {
      return false;
    }
    return p.is_global;
  }
  
  public ArrayList<String> getParamsWithDefaults() {
    return (ArrayList<String> )this.params_with_default_list.clone();
  }
  
  private class Param
  {

    public String default_value = null;
    public boolean save = true;
    public boolean sensitive = false;
    public boolean is_global = false;
    
    public Param(String default_value, boolean save)
    {
      this.default_value = default_value;
      this.save = save;
    }
    
    public Param(String default_value, boolean save, boolean sensitive) {
      this.default_value = default_value;
      this.save = save;
      this.sensitive = sensitive;
    }

    public Param(String default_value, boolean save, boolean sensitive, boolean global) {
      this.default_value = default_value;
      this.save = save;
      this.sensitive = sensitive;
      this.is_global = global;
    }
    
  }
}
