/**
 *#########################################################################
 *
 * A component of the Gatherer application, part of the Greenstone digital
 * library suite from the New Zealand Digital Library Project at the
 * University of Waikato, New Zealand.
 *
 * Author: Katherine Don, Greenstone Digital Library, University of Waikato
 *
 * Copyright (C) 2005 New Zealand Digital Library Project
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *########################################################################
 */
package org.greenstone.gatherer.gui;

import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.event.*;
import javax.swing.text.*;
import java.io.*;
import java.util.ArrayList;

import org.w3c.dom.Document;

import org.greenstone.gatherer.Dictionary;
import org.greenstone.gatherer.Configuration;
import org.greenstone.gatherer.Gatherer;
import org.greenstone.gatherer.cdm.Argument;
import org.greenstone.gatherer.cdm.ArgumentControl;
import org.greenstone.gatherer.cdm.CollectionDesignManager;
import org.greenstone.gatherer.cdm.Plugin;
import org.greenstone.gatherer.collection.CollectionManager;
import org.greenstone.gatherer.collection.ScriptOptions;
import org.greenstone.gatherer.collection.Collection;
import org.greenstone.gatherer.feedback.Base64;
import org.greenstone.gatherer.greenstone.LocalGreenstone;
import org.greenstone.gatherer.gui.TestingPreparation;
import org.greenstone.gatherer.gui.tree.DragTree;
import org.greenstone.gatherer.metadata.MetadataXMLFileManager;
import org.greenstone.gatherer.metadata.MetadataElement;
import org.greenstone.gatherer.metadata.MetadataSet;
import org.greenstone.gatherer.metadata.MetadataSetManager;
import org.greenstone.gatherer.metadata.MetadataTools;
import org.greenstone.gatherer.remote.RemoteGreenstoneServer;
import org.greenstone.gatherer.shell.GShell;
import org.greenstone.gatherer.shell.GShellEvent;
import org.greenstone.gatherer.shell.GShellListener;
import org.greenstone.gatherer.util.Utility;
import org.greenstone.gatherer.util.XMLTools;

public class ExplodeMetadataDatabasePrompt	
    extends ModalDialog
    implements GShellListener
{
    /** The size of this new collection dialog box. */
    static private Dimension SIZE = new Dimension(675, 350);
    private JDialog self;
    
    /** the file we wil be exploding */
    private File metadata_file = null;
    /** the list of potential plugins to be used */
    private Argument plugin_arg = null;
    /** holds all the available options for the exploding script */
    private ScriptOptions options = null;
    /** the pane containing the options */
    private JPanel options_pane = null;
    /** the error message if any */
    private StringBuffer error_message = null;
    /** whether we were successful or not */
    private boolean successful;


    public ExplodeMetadataDatabasePrompt(File source_file) {
	super(Gatherer.g_man, true);
	this.self = this;
	this.metadata_file = source_file;
        this.setComponentOrientation(Dictionary.getOrientation());
	// check that we actually have an explodable file
	ArrayList exp_plugins = CollectionDesignManager.plugin_manager.getExploderPlugins(source_file);
	if (exp_plugins.size() == 0) {
	    JOptionPane.showMessageDialog(Gatherer.g_man, Dictionary.get("ExplodeMetadataPrompt.NotExplodable"), Dictionary.get("ExplodeMetadataPrompt.Title"), JOptionPane.ERROR_MESSAGE);
	    return;
	}
	plugin_arg = createPluginArgument(exp_plugins);
	setJMenuBar(new SimpleMenuBar("explodingmetadata"));
	setSize(SIZE);
	setTitle(Dictionary.get("ExplodeMetadataPrompt.Title"));
	
	// set up the script options
	// we have empty initial values
	String dom_string = "<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n<Options/>";
	Document doc = XMLTools.parseXML(new StringReader(dom_string));
	options = new ScriptOptions(doc.getDocumentElement(), "explode_metadata_database.pl");
	
	// Creation
	JPanel content_pane = (JPanel) getContentPane();
        content_pane.setComponentOrientation(Dictionary.getOrientation());
	content_pane.setOpaque(true);
	
	options_pane = new JPanel();
        options_pane.setComponentOrientation(Dictionary.getOrientation());
	addScriptOptions(options_pane);
	JScrollPane middle_pane = new JScrollPane(options_pane);
        middle_pane.setComponentOrientation(Dictionary.getOrientation());
        
	JTextArea instructions_area = new JTextArea(Dictionary.get("ExplodeMetadataPrompt.Instructions"));
	instructions_area.setComponentOrientation(Dictionary.getOrientation());
        instructions_area.setEditable(false);
	instructions_area.setLineWrap(true);
	instructions_area.setRows(5);
	instructions_area.setWrapStyleWord(true);
	
	JPanel button_pane = new JPanel();
	button_pane.setComponentOrientation(Dictionary.getOrientation());
        JButton explode_button = new GLIButton(Dictionary.get("ExplodeMetadataPrompt.Explode"), Dictionary.get("ExplodeMetadataPrompt.Explode_Tooltip"));
	JButton cancel_button = new GLIButton(Dictionary.get("General.Cancel"), Dictionary.get("General.Cancel_Tooltip"));
	
	// Connection
	cancel_button.addActionListener(new CancelListener());
	explode_button.addActionListener(new ExplodeListener());

	// Layout

	button_pane.setBorder(BorderFactory.createEmptyBorder(5,0,0,0));
	button_pane.setLayout(new GridLayout(1,2));
	button_pane.add(explode_button);
	button_pane.add(cancel_button);

	content_pane.setBorder(BorderFactory.createEmptyBorder(5,5,5,5));
	content_pane.setLayout(new BorderLayout());
	content_pane.add(instructions_area, BorderLayout.NORTH);
	content_pane.add(middle_pane, BorderLayout.CENTER);
	content_pane.add(button_pane, BorderLayout.SOUTH);

	// Final dialog setup & positioning.
	Dimension screen_size = Configuration.screen_size;
	setLocation((screen_size.width - SIZE.width) / 2, (screen_size.height - SIZE.height) / 2);

	TestingPreparation.setNamesRecursively(this);
	setVisible(true);
    }

    public void destroy() 
    {
       	
    }

    /** All implementation of GShellListener must include this method so the listener can be informed of messages from the GShell.
     * @param event A <strong>GShellEvent</strong> that contains, amoung other things, the message.
     */
    public synchronized void message(GShellEvent event) {
	String message = event.getMessage();
	if (message.startsWith("explode_metadata_database.pl>")) {
	    message = message.substring(29);
	    error_message.append(message);
	    error_message.append("\n");
	}
    }

    /** All implementation of GShellListener must include this method so the listener can be informed when a GShell begins its task. Implementation side-effect, not actually used.
     * @param event A <strong>GShellEvent</strong> that contains details of the initial state of the <strong>GShell</strong> before task comencement.
     */
    public synchronized void processBegun(GShellEvent event) {
	// We don't care. 
    }
    
    /** All implementation of GShellListener must include this method so the listener can be informed when a GShell completes its task.
     * @param event A <strong>GShellEvent</strong> that contains details of the final state of the <strong>GShell</strong> after task completion.
     */
    public synchronized void processComplete(GShellEvent event) {
	successful = false;
	if(event.getStatus() == GShell.OK) {
	    successful = true;
	}
    }

    private Argument createPluginArgument(ArrayList plugin_list) {
	Argument arg = new Argument();
	arg.setName("plugin");
	arg.setDescription(Dictionary.get("ExplodeMetadataPrompt.Plugin"));
	arg.setRequired(true);
	arg.setType(Argument.ENUM);
	for (int i=0; i<plugin_list.size(); i++) {
	    Plugin p = (Plugin)plugin_list.get(i);
	    arg.addOption(p.getName(), p.getName());
	}
	return arg;
    }
    private void addScriptOptions(JPanel options_pane) {

	options_pane.setBorder(BorderFactory.createEmptyBorder(5,5,5,5));
	options_pane.setBackground(Configuration.getColor("coloring.collection_heading_background", false));
	options_pane.setLayout(new BoxLayout(options_pane, BoxLayout.Y_AXIS));

	int current_mode = Configuration.getMode();
	int total_argument_count = options.getArgumentCount();
       
	// first we add the plugin arg
	ArgumentControl argument_control = new ArgumentControl(plugin_arg, true, null);
	argument_control.setBackground(Configuration.getColor("coloring.collection_heading_background", false));
	options_pane.add((JComponent)argument_control);
	for(int i = 0; i < total_argument_count; i++) {
	    // Retrieve the argument so we know how to format the control.
	    Argument argument = options.getArgument(i);
	    if(!argument.isHiddenGLI() && argument.getModeLevel() <= current_mode) {
		if (argument.getName().equals("metadata_set")) {
		    argument.setType(Argument.METADATA_SET_NAMESPACE);
		    argument_control = new ArgumentControl(argument, true, "Exploded Metadata Set (exp)");
		}
		else {
		    // by default, all args are disabled, and no value
		    argument_control = new ArgumentControl(argument, false, null);
		}
		// make sure they are coloured the way we want - this is not the standard arg control coloring
		argument_control.setBackground(Configuration.getColor("coloring.collection_heading_background", false));
		options_pane.add((JComponent)argument_control);
	    }
	}
    }

    private void updateScriptOptions() {
	for(int i = 0; i < options_pane.getComponentCount(); i++) {
	    Component component = options_pane.getComponent(i);
	    if(component instanceof ArgumentControl) {
		ArgumentControl ac = (ArgumentControl)component;
		String name = ac.getArgumentName();
		String value = ac.getValue();
		boolean enabled = ac.isEnabled();
		if (!enabled && value==null) {
		    // flag type, remove from options altogether
		    options.removeValue(name);
		} else {
		    if (value.equals("")) {
			// if the value is empty, we don't want it to be passed to the script
			options.setValue(name, false, value);
		    } else {
			options.setValue(name, enabled, value);
		    }
		}

	    }
	}
    }

    private int explodeMetadata()
    {
	// Generate the explode_metadata_database.pl command
	ArrayList command_parts_list = new ArrayList();
	if (!Gatherer.isGsdlRemote) {
	    command_parts_list.add(Configuration.perl_path);
	    command_parts_list.add("-S");
	}
	command_parts_list.add(LocalGreenstone.getBinScriptDirectoryPath() + "explode_metadata_database.pl");
	// Add in the options we preset in GLI
	if (Gatherer.GS3) {
	    command_parts_list.add("-site");
	    command_parts_list.add(Configuration.site_name);
	}
	command_parts_list.add("-collection");
	command_parts_list.add(Gatherer.c_man.getCollection ().getName ());
	command_parts_list.add("-use_collection_plugin_options");
	// Add in all the options from the user
	String[] explode_options = options.getValues();
	for (int i = 0; i < explode_options.length; i++) {
	    command_parts_list.add(explode_options[i]);
	}

	// Local case
	if (!Gatherer.isGsdlRemote) {
	    // Add in the filename
	    command_parts_list.add(metadata_file.getPath());
	}
	// Remote case
	else {
	    // Add in the filename, relative to the collection directory
	    String collection_name = CollectionManager.getLoadedCollectionName();
	    String collection_directory_path = CollectionManager.getCollectionDirectoryPath(collection_name);
	    String metadata_file_relative_path = Gatherer.remoteGreenstoneServer.getPathRelativeToDirectory(
                                                          metadata_file, collection_directory_path);
	    metadata_file_relative_path = metadata_file_relative_path.replace(File.separatorChar, '|');
	    command_parts_list.add("-file");
	    command_parts_list.add(Base64.encodeBytes(metadata_file_relative_path.getBytes()));

	    // When running remotely we also need the collection name as the last argument
	    command_parts_list.add(collection_name);
	}

	// Run the explode_metadata_database.pl command
	String[] command_parts = (String[]) command_parts_list.toArray(new String[0]);
	this.error_message = new StringBuffer();
	GShell process = new GShell(command_parts, GShell.EXPLODE, 3, this, null, GShell.GSHELL_EXPLODE);
	//process.start();
	process.run();
	
	if (successful) {
	    return 0;
	} 
	return -1;
    }

    private void resultPrompt(boolean success, String message)
    {
     Gatherer.invokeInEDT_replacesProceedInCurrThread(
      "ExplodeMetadataDatabasePrompt.resultPrompt()",
      Gatherer.SYNC,
      new Runnable() {
       public void run() {
	if (success) {
	    // !!! TO DO: Get explode_metadata_database.pl strings translated and use SimpleResultDialog below
	    JOptionPane.showMessageDialog(null, Dictionary.get("ExplodeMetadataPrompt.Successful_Explode", metadata_file.getName()), Dictionary.get("ExplodeMetadataPrompt.Successful_Title"), JOptionPane.INFORMATION_MESSAGE);
	}
	else {
	    String title = Dictionary.get("ExplodeMetadataPrompt.Failed_Title");
	    String label = Dictionary.get("ExplodeMetadataPrompt.Failed_Explode", metadata_file.getName());
	    SimpleResultDialog result_dialog = new SimpleResultDialog(title, label, message);
	    TestingPreparation.setNamesRecursively(result_dialog);
	    result_dialog.setVisible(true); // Blocks
	    result_dialog.dispose();
	    result_dialog = null;
	}
       } // end run
      }); // end invoke on EDT
    }

    private class CancelListener
	implements ActionListener {
	public void actionPerformed(ActionEvent event) {
	    self.dispose();
	}
    }
    
    private class ExplodeListener
	implements ActionListener {
	
	public void actionPerformed(ActionEvent event) {
	    Gatherer.g_man.wait(true);
	    // update the options
	    updateScriptOptions();
	    self.dispose();  
	    (new ExplodeMetadataDatabaseTask()).start();
	}
    }


    private class ExplodeMetadataDatabaseTask
	extends Thread
    {
	public ExplodeMetadataDatabaseTask()
	{
	}

	public void run()
	{
	    int exit_value = explodeMetadata();
	    if (exit_value == 0) { // success
		// Clear out the old directory containing the metadata file and download the new directory
		if (Gatherer.isGsdlRemote) {
		    Utility.delete(metadata_file.getParentFile());
		    Gatherer.remoteGreenstoneServer.downloadCollectionFile(
                             CollectionManager.getLoadedCollectionName(), metadata_file.getParentFile());
		}

		// Load the new metadata.xml files
                Collection collection = Gatherer.c_man.getCollection();
             	MetadataXMLFileManager.loadMetadataXMLFiles(metadata_file.getParentFile(),collection.toSkimFile());
		Gatherer.g_man.refreshCollectionTree(DragTree.COLLECTION_CONTENTS_CHANGED);
		resultPrompt(true, error_message.toString());
	    } else { // failure
		resultPrompt(false, error_message.toString());
	    }
	    error_message = null;
	    Gatherer.g_man.wait(false);
	}
    }    
}
