###########################################################################
#
# scriptutil.pm -- various useful utilities for the scripts
# A component of the Greenstone digital library software
# from the New Zealand Digital Library Project at the 
# University of Waikato, New Zealand.
#
# Copyright (C) 1999 New Zealand Digital Library Project
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#
###########################################################################

package scriptutil;

use strict;
no strict 'subs'; # allow barewords (eg STDERR) as function arguments
no strict 'refs'; # ...but allow filehandles to be variables and vice versa

use gsprintf 'gsprintf';

# returns ($removeold, $keepold, $incremental, $incremental_mode)
sub check_removeold_and_keepold {

    my ($removeold, $keepold, $incremental, $dir, $collectcfg) = @_;    

    if ($keepold && $removeold) {
	gsprintf(STDERR, "{scripts.only_one_old_option}\n");
	exit(2)	
    } 

    if ($incremental && $removeold) {
	gsprintf(STDERR, "{scripts.inc_remove_conflict}\n", $dir);
	sleep(3); #just in case
	return (1,0,0,"none"); 
	
    } 

    
    # Incremental mode may be set to "none", "onlyadd" or "all"
    # depending on status of -keepold and -incremental flags
    my $incremental_mode = "none";
    if ($incremental) {
	$incremental_mode = "all";
    } elsif ($keepold) {
	$incremental_mode = "onlyadd";
    }

    if (!$keepold && !$removeold && !$incremental && defined $collectcfg) {
	# we only look at config file options if we dont have these on the command line
	if (defined $collectcfg->{'removeold'} && $collectcfg->{'removeold'} =~ /^true$/i ) {
	    $removeold = 1;
	} elsif (defined $collectcfg->{'keepold'} && $collectcfg->{'keepold'} =~ /^true$/i) {
	    $keepold = 1;
	    $incremental_mode = "onlyadd";
	} elsif (defined $collectcfg->{'incremental'} && $collectcfg->{'incremental'} =~ /^true$/i) {
	    $incremental = 1;
	    $incremental_mode = "all";
	}
    }

    if (!$keepold && !$removeold && !$incremental) {
	gsprintf(STDERR, "{scripts.no_old_options} \n", $dir);
	sleep(3); #just in case
	return (1,0,0,"none");
    }
    
    # incremental implies keepold
    if ($incremental) {
	$keepold = 1;
    }
    return ($removeold, $keepold, $incremental, $incremental_mode);

}


# returns ($removeold, $keepold, $replaceold, $incremental, $incremental_mode)
sub check_removeold_keepold_replaceold {

    my ($removeold, $keepold, $replaceold, $incremental, $dir, $collectcfg) = @_;    

    my $old_count = 0;
    $old_count++ if $removeold;
    $old_count++ if $keepold;
    $old_count++ if $replaceold;

    if ($old_count>1) {
	gsprintf(STDERR, "{scripts.only_one_old_option}\n");
	exit(2);
    } 

    if (($incremental && $removeold) ) {
	gsprintf(STDERR, "{scripts.inc_remove_conflict}\n", $dir);
	sleep(5); #just in case
	return (1,0,0,0,"none"); 	
    } 

    # Determine what the internal 'incremental_mode' is:
    # => May be set to "none", "onlyadd" or "all"
    # Based on status of (-keepold|-removeold) and -incremental flags
    #
    # With the introduction of file-level document-version (fldv) history, the chosen name 'onlyadd'
    # for when '-keepold' is on is a bit misleading. However, it does still get us "over the line"
    # in terms of how it functionally operates.  In the case where pre-existing content is
    # still in the 'import' folder, then when everything in 'archives_keepold' gets copied
    # back, any pre-existing documents from import (which will have resulted in a doc folder
    # in 'archives') will trigger a file-level document-version history folder inside it.  For any
    # content that was new in 'import', it won't have a pre-existing folder inside 'archives'
    # and so will appear as a new folder with *no* file-level document-version history folder
    # inside it (effectively why the keepold incremental mode was originally called 'onlyadd').

    my $incremental_mode = "none";
    if ($incremental) {
	$incremental_mode = "all";
    } elsif ($keepold || $replaceold) {
	$incremental_mode = "onlyadd";
    }

    if (!$keepold && !$removeold && !$replaceold && !$incremental && defined $collectcfg) {
	# we only look at config file options if we don't have these on the command line
	if (defined $collectcfg->{'removeold'} && $collectcfg->{'removeold'} =~ /^true$/i ) {
	    $removeold = 1;
	} elsif (defined $collectcfg->{'keepold'} && $collectcfg->{'keepold'} =~ /^true$/i) {
	    $keepold = 1;
	    $incremental_mode = "onlyadd";
	} elsif (defined $collectcfg->{'replaceold'} && $collectcfg->{'replaceold'} =~ /^true$/i) {
	    $replaceold = 1;
	    $incremental_mode = "onlyadd";
	} elsif (defined $collectcfg->{'incremental'} && $collectcfg->{'incremental'} =~ /^true$/i) {
	    $incremental = 1;
	    $incremental_mode = "all";
	}

	# Go through the same checks as before
	my $cfg_old_count = 0;
	$cfg_old_count++ if $removeold;
	$cfg_old_count++ if $keepold;
	$cfg_old_count++ if $replaceold;

	if ($cfg_old_count>1) {
	    gsprintf(STDERR, "{scripts.only_one_old_option}\n");
	    exit(2);
	}

	if (($incremental && $removeold) ) {
	    gsprintf(STDERR, "{scripts.inc_remove_conflict}\n", $dir);
	    sleep(5); #just in case
	    return (1,0,0,0,"none"); 	
	} 
    }

    # default to -removeold if nothing specified
    if (!$keepold && !$removeold && !$replaceold && !$incremental) {
	gsprintf(STDERR, "{scripts.no_old_options} \n", $dir);
	sleep(5); #just in case
	return (1,0,0,0,"none");
    }
    
    # incremental implies keepold
    if ($incremental) {
	if (!$replaceold) {
	    $keepold = 1;
	}
    }
    return ($removeold, $keepold, $replaceold, $incremental, $incremental_mode);

}



1;
