/**
 *############################################################################
 * A component of the Greenstone Librarian Interface, part of the Greenstone
 * digital library suite from the New Zealand Digital Library Project at the
 * University of Waikato, New Zealand.
 *
 * Author: Michael Dewsnip, NZDL Project, University of Waikato, NZ
 *
 * Copyright (C) 2004 New Zealand Digital Library Project
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *############################################################################
 */

package org.greenstone.gatherer.metadata;


import java.io.*;


/** This class represents one metadata (element, value) pair */
public class MetadataValue
    implements Comparable
{
    /** The folder that this piece of metadata was inherited from, if applicable */
    private File folder_metadata_inherited_from = null;
    private MetadataElement metadata_element = null;
    private MetadataValueTreeNode metadata_value_tree_node = null;
  // this will be set to true if the metadata element definition says accumulating = false.
  private boolean must_not_accumulate = false;
    private boolean is_accumulating_metadata = false;
    /** Whether this metadata applies to only one file (ie. there are no wildcards in the FileSet value) */
    private boolean is_one_file_only_metadata = true;


    public MetadataValue(MetadataElement metadata_element, MetadataValueTreeNode metadata_value_tree_node)
    {
	this.metadata_element = metadata_element;
	this.metadata_value_tree_node = metadata_value_tree_node;
	if (metadata_element != null) {
	  if (!metadata_element.isAccumulating()) {
	    must_not_accumulate = true;
	  }
	}
    }


    public int compareTo(Object metadata_value_object)
    {
	MetadataValue metadata_value = (MetadataValue) metadata_value_object;

	// First compare the metadata elements
	int c = MetadataSetManager.compareMetadataElements(this.getMetadataElement(), metadata_value.getMetadataElement());
	if (c != 0) {
	    return c;
	}

	// Then use the metadata values
	return this.getFullValue().compareTo(metadata_value.getFullValue());
    }


    public File getFolderMetadataInheritedFrom()
    {
	return folder_metadata_inherited_from;
    }


    public MetadataElement getMetadataElement()
    {
	return metadata_element;
    }


    public String getFullValue()
    {
	return metadata_value_tree_node.getFullValue();
    }


    public String getValue()
    {
	return metadata_value_tree_node.getValue();
    }


    public MetadataValueTreeNode getMetadataValueTreeNode()
    {
	return metadata_value_tree_node;
    }


    public void inheritsMetadataFromFolder(File folder_metadata_inherited_from)
    {
	this.folder_metadata_inherited_from = folder_metadata_inherited_from;
    }


    public boolean isAccumulatingMetadata()
    {
	return is_accumulating_metadata;
    }


    public boolean isInheritedMetadata()
    {
	return (folder_metadata_inherited_from != null);
    }


    public boolean isOneFileOnlyMetadata()
    {
	return is_one_file_only_metadata;
    }


    public void setIsAccumulatingMetadata(boolean is_accumulating_metadata)
    {
      if (!must_not_accumulate) {
	this.is_accumulating_metadata = is_accumulating_metadata;
      } 
    }


    public void setIsOneFileOnlyMetadata(boolean is_one_file_only_metadata)
    {
	this.is_one_file_only_metadata = is_one_file_only_metadata;
    }
}
