/**
 *#########################################################################
 *
 * A component of the Gatherer application, part of the Greenstone digital
 * library suite from the New Zealand Digital Library Project at the
 * University of Waikato, New Zealand.
 *
 * Author: Xiao F. Yu, Greenstone Digital Library, University of Waikato
 *
 * Copyright (C) 1999 New Zealand Digital Library Project
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *########################################################################
 */
package org.greenstone.gatherer.cdm;

import org.greenstone.gatherer.DebugStream;
import org.greenstone.gatherer.util.Codec;
import org.greenstone.gatherer.util.StaticStrings;
import org.greenstone.gatherer.util.XMLTools;
import org.w3c.dom.*;
import org.greenstone.gatherer.metadata.MetadataElement;
import org.greenstone.gatherer.metadata.MetadataTools;

/**
 * This class is the greenstone 3 version of the Format.java in gs2 for handling
 * 'gsf' format statements.
 */
public class Format4gs3 implements DOMProxyListEntry
{

	/** The element this format is based on. */
	private Element element = null;
	/** We keep a copy of the feature name */
	private String feature_name = null;
	/**
	 * The format statement for this feature (also contains classifier options
	 * if its a classifier), used by the format list on the top in GLI.
	 */
	private String feature_format = null;

	/**
	 * The format statement for this feature (does not contain options in the
	 * front), displayed in the format editor at the bottom in GLI. If this
	 * format is not a classifier, feature_format==pure_format
	 */
	private String pure_format = null;

	/**
	 * Wether this feature is about a classifier. The other way to check this is
	 * to see if feature_name starts with "CL"
	 */
	private static boolean is_classifier = false;
	/** If this format is about a classifier, this contains all its options */
	private String classifier_options = "";

	/** Constructor only to be used during DOMProxyListModel initialization. */
	public Format4gs3()
	{

	}

	/**
	 * Constructor for a format-string format command.
	 * 
	 * @param element
	 *            The <Strong>element</strong> this format object contains.
	 */
	public Format4gs3(Element element)
	{
		this.element = element;

		is_classifier = element.getParentNode().getNodeName().equals(StaticStrings.CLASSIFY_ELEMENT);

		if (is_classifier && CollectionDesignManager.classifier_manager != null)
		{
			//The getClassifierPosition() method also sets the classifier_options variable.
			feature_name = Classifier.CLASSIFIER_PREFIX + getClassifierPosition(element);
			Classifier classifier = getClassifier(feature_name);
			//set the classifier_options string
			if (classifier != null)
			{
				classifier_options = classifier.toString();
			}

			pure_format = toOneLineFormat(XMLTools.getNodeText(element));
			feature_format = classifier_options + " " + pure_format;
		}
		else
		{
			feature_name = element.getAttribute(StaticStrings.NAME_ATTRIBUTE);
			feature_format = toOneLineFormat(XMLTools.getNodeText(element));
			pure_format = feature_format;
		}

	}

	//
	public Format4gs3(String feature_name, String feature_format)
	{
		this.feature_name = feature_name;
		this.feature_format = feature_format;
		this.pure_format = feature_format;

		is_classifier = feature_name.startsWith(Classifier.CLASSIFIER_PREFIX);

		element = CollectionConfiguration.createElement(StaticStrings.FORMAT_STR);
		if (is_classifier)
		{

			//Now get the options of the classifier
			Classifier classifier = getClassifier(feature_name);
			//set the classifier_options string
			classifier_options = classifier.toString();

			feature_format = classifier_options + " " + feature_format;
		}
		else
		{
			element.setAttribute(StaticStrings.NAME_ATTRIBUTE, feature_name);
		}

		XMLTools.setNodeText(element, toOneLineFormat(pure_format));

	}

	public int getClassifierPosition(Element element)
	{
		//we could also here use 'element.getParentNode().getParentNode()' to get this root node
		Element root = CollectionDesignManager.collect_config.getDocumentElement();
		NodeList classify_list = root.getElementsByTagName(StaticStrings.CLASSIFY_ELEMENT);
		int num_classify_elements = classify_list.getLength();
		for (int i = 0; i < num_classify_elements; i++)
		{
			Element classify = (Element) classify_list.item(i);
			Element format = (Element) XMLTools.getChildByTagName(classify, StaticStrings.FORMAT_STR);
			if (format == element)
			{
				//Now get the options of the classifier
				Classifier classifier = CollectionDesignManager.classifier_manager.getClassifier(i);
				classifier_options = classifier.toString();
				// i+1 because java is from 0 up, greenstone starts from 1.
				return i + 1;
			}
		}
		return -1;
	}

	public static Classifier getClassifier(String feature_name)
	{
		int index = Integer.parseInt(feature_name.substring(Classifier.CLASSIFIER_PREFIX.length()));
		Classifier classifier = CollectionDesignManager.classifier_manager.getClassifier(index - 1);
		return classifier;
	}

	//This method is used when the add button is clicked and a new format element for a classifier is added to
	// the DOM tree, we need to figure out which Classify element to append this format element to.
	public Element getClassifyElement()
	{

		//String name = classifier.getPositionString();
		int classifier_index = Integer.parseInt(feature_name.substring(Classifier.CLASSIFIER_PREFIX.length()));
		Element root = CollectionDesignManager.collect_config.getDocumentElement();
		NodeList classify_list = root.getElementsByTagName(StaticStrings.CLASSIFY_ELEMENT);

		// - 1 because java is from 0 up, greenstone starts from 1.
		classifier_index = classifier_index - 1;
		return (Element) classify_list.item(classifier_index);
	}

	/**
	 * Method to translate this class information into a line of text as you
	 * would expect on the format list panel
	 * 
	 * @return A <strong>String</strong> containing the format command.
	 */
	public String toString()
	{

		return feature_name + " " + toOneLineFormat(feature_format);
	}

	public void update()
	{

		if (is_classifier)
		{
			feature_name = Classifier.CLASSIFIER_PREFIX + getClassifierPosition(element);
			Classifier classifier = getClassifier(feature_name);
			//update the classifier_options string
			classifier_options = classifier.toString();
			pure_format = toOneLineFormat(XMLTools.getNodeText(element));
			feature_format = classifier_options + " " + pure_format;
		}
	}

	public String getFormatedFormat()
	{
		return toFormatedFormat(feature_format);
	}

	public String getPureFormat()
	{
		return toFormatedFormat(pure_format);
	}

	public Classifier getClassifier()
	{
		if (!isClassifier())
		{

			return null;
		}
		int index = Integer.parseInt(feature_name.substring(Classifier.CLASSIFIER_PREFIX.length()));
		//index-1 because java indexes from 0, but greenstone is from 1.
		index = index - 1;
		return CollectionDesignManager.classifier_manager.getClassifier(index);
	}

	public void setFeatureName(String name)
	{
		feature_name = name;
	}

	/**
	 * Set the value of value. Hmmm.
	 * 
	 * @param value
	 *            The new value from value as a <strong>String</strong>.
	 */
	public void setFeatureFormat(String value)
	{
		if (is_classifier)
		{
			feature_format = classifier_options + " " + value;
		}
		else
		{
			feature_format = value;
		}
	}

	public void setPureFormat(String value)
	{
		pure_format = value;
		setFeatureFormat(value);
		XMLTools.setNodeText(element, pure_format);
	}

	public static String toOneLineFormat(String str)
	{
		
		return str.replaceAll("\n", "").replaceAll("\\s+", " ").replaceAll(" <", "<");
	}

	public static String toFormatedFormat(String str)
	{
		return str.replaceAll("\\s+", " ").replaceAll("<", "\n<").replaceFirst("\n", "");
	}

	public int compareTo(Object object)
	{
		if (object == null)
		{ // It seems that occasionally compareTo is called and somehow null ends up in comparison.
			return -1;
		}
		return this.getFeatureName().compareToIgnoreCase(((Format4gs3) object).getFeatureName());
	}

	public DOMProxyListEntry create(Element element)
	{
		return new Format4gs3(element);
	}

	public boolean equals(Object object)
	{
		return (compareTo(object) == 0);
	}

	public Element getElement()
	{
		return element;
	}

	public boolean isClassifier()
	{
		return feature_name.startsWith(Classifier.CLASSIFIER_PREFIX);
	}

	public String getFeatureName()
	{
		return feature_name;
	}

	//The format appended as a row of string used to display in the format list panel
	public String getFeatureFormat()
	{
		return feature_format;
	}

	public void setElement(Element element)
	{
		this.element = element;
	}

	public boolean isAssigned()
	{
		return true;
	}

	public void setAssigned(boolean assigned)
	{
	}

}