###########################################################################
#
# rssinfo.pm -- handles writing to the rss file/db
# A component of the Greenstone digital library software
# from the New Zealand Digital Library Project at the 
# University of Waikato, New Zealand.
#
# Copyright (C) 2025 New Zealand Digital Library Project
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#
###########################################################################

package rssinfo;

use strict;
no strict 'subs';
no strict 'refs';

use dbutil;
use FileUtils;
use util;

sub save_rss_entry {

    my ($oid, $title, $infodbtype, $output_dir) = @_;

    my $rss_entry = "<item>\n";
    $rss_entry   .= "   <title>$title</title>\n";
    if(&util::is_gs3()) {
        $rss_entry   .= "   <link>_httpdomain__httpcollection_/document/$oid</link>\n";
    } else {
        $rss_entry   .= "   <link>_httpdomainHtmlsafe__httpcollection_/document/$oid</link>\n";
    }
    $rss_entry   .= "</item>";

    # NB no current drivers support RSS
    if (defined(&dbutil::supportsRSS) && &dbutil::supportsRSS($infodbtype))
    {
        my $rss_db = &dbutil::get_infodb_file_path($infodbtype, 'rss-items', $output_dir);
        my $rss_db_fh = &dbutil::open_infodb_write_handle($infodbtype, $rss_db, 'append');
        &dbutil::write_infodb_rawentry($infodbtype, $rss_db_fh, $oid, $rss_entry);
        &dbutil::close_infodb_write_handle($infodbtype, $rss_db_fh);
    }
    else
    {
        my $rss_filename = &FileUtils::filenameConcatenate($output_dir,"rss-items.rdf");
        my $rss_fh;
        if (&FileUtils::openFileHandle($rss_filename, '>>', \$rss_fh, "utf8"))
        {
	    print $rss_fh $rss_entry . "\n";
	    &FileUtils::closeFileHandle($rss_filename, \$rss_fh);
        }
        else
        {
            print STDERR "Error: Failed to open $rss_filename\n$!\n";
        }
    }
}

sub delete_rss_entry {

    my ($oid, $infodbtype, $output_dir) = @_;

    my $rss_filename = &FileUtils::filenameConcatenate($output_dir,"rss-items.rdf");

    my $rss_contents = &util::read_utf8_textfile($rss_filename);
    
    my $id_line;
    if (&util::is_gs3()) {
        $id_line = "<link>_httpdomain__httpcollection_/document/$oid</link>";
    } else {
        $id_line = "<link>_httpdomainHtmlsafe__httpcollection_/document/$oid</link>\n";
    }

    my $full_item = "<item>\\s+<title>[^<]*</title>\\s+$id_line\\s+</item>";
    
    if ($rss_contents =~ s#$full_item##s) {
        # we deleted soemthing - write out the contents
        my $rss_fh;
        &FileUtils::openFileHandle($rss_filename, 'w', \$rss_fh, "utf8");
        print $rss_fh "$rss_contents\n";
        &FileUtils::closeFileHandle($rss_filename, \$rss_fh);
        
    }

    
}

1;

