/**
 *#########################################################################
 *
 * A component of the Gatherer application, part of the Greenstone digital
 * library suite from the New Zealand Digital Library Project at the
 * University of Waikato, New Zealand.
 *
 * Author: John Thompson, Greenstone Digital Library, University of Waikato
 *
 * Copyright (C) 1999 New Zealand Digital Library Project
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *########################################################################
 */
package org.greenstone.gatherer.cdm;

import java.io.*;
import java.util.*;
import org.greenstone.gatherer.DebugStream;
import org.greenstone.gatherer.greenstone.Plugins;
import org.greenstone.gatherer.util.StaticStrings;
import org.w3c.dom.*;


/** This class is responsible for storing information from a parsed pluginfo call in such a way that it allows easy access to parsed details for the purposes of user design and specification of plugins. */
public class Plugin
    extends ArgumentContainer
{
    private String default_block_expression = "";
    private String default_process_expression = "";
    private boolean does_explode_metadata_databases = false; 
    private boolean does_replace_srcdocs_with_html = false; // to work with replace_srcdoc_with_html.pl
    /** Plugins are loaded as needed. This variable indicates 1. whether a plugin has been loaded already; 2. whether a 
     * plugin has been successfully loaded. When both are true, has_loaded_options will be true. */
    private boolean has_loaded_options = false;

    /** Constructor used in DOMProxyListModel initializations, and Library Level. Used for Base plugins (those in the list of available plugins, not ones that are in the DOMProxyList)
     */
    public Plugin() {
    }

    
    public Plugin(Element element, Plugin base_plugin) {
	super(element, base_plugin);
    }


    /** Method to compare two plugins for ordering.
     * We override the base method cos we compare by plugin name rather than toString()
     */
    public int compareTo(Object object) {
	if(object instanceof Plugin) {
	    return name.compareTo(((Plugin)object).getName());
	}
	return -1;
    }

    
    /** The assigned plugin constructor.
     * @param element the DOM Element this plugin is based upon
     */
    public DOMProxyListEntry create(Element element) {
	String plugin_name = element.getAttribute(StaticStrings.TYPE_ATTRIBUTE);
	// Determine the base plugin from the plugin name
	Plugin base_plugin = Plugins.getPlugin(plugin_name, true);
	Plugin plugin = new Plugin(element, base_plugin);
	if (base_plugin == null) {
	    plugin.setAssigned(false);
	}
	base_plugin = null;
	plugin_name = null;
	return plugin;
    }


    public boolean hasLoadedOptions()
    {
	return has_loaded_options;
    }


    /** Checks whether the plugin this instance is based on processes metadata databases that can be exploded. */
    public boolean doesExplodeMetadataDatabases()
    {
	Plugin base_plugin = Plugins.getPlugin(getName(), false);
	if (base_plugin == null) {
	    return false;
	}
	return base_plugin.does_explode_metadata_databases;
    }

    /** Checks whether the plugin this instance is based on processes source documents that can be replaced with their Greenstone generated html variants. 
     * This method works with replace_srcdoc_with_html.pl */
    public boolean doesReplaceSrcDocsWithHtml()
    {
	//return does_replace_srcdocs_with_html;
	Plugin base_plugin = Plugins.getPlugin(getName(), false);
	if (base_plugin == null) {
	  return false;
	}
	return base_plugin.does_replace_srcdocs_with_html;
    }

    /** Checks whether this plugin instance will process the specified file (given its block_exp). */
    public boolean doesBlockFile(File file)
    {
	// Check the filename against the plugin's block_exp value
	ArrayList arguments = getArguments();
	for (int i = 0; i < arguments.size(); i++) {
	    Argument argument = (Argument) arguments.get(i);
	    if (argument.getName().equals("block_exp")) {
		// Try the assigned value first, for when the user has manually set the value
		String regular_expression = argument.getValue();
		if (regular_expression == null || regular_expression.equals("")) {
		    // Not set, so use the default value
		    regular_expression = argument.getDefaultValue();
		    if (regular_expression.equals("")) {
			continue;
		    }
		}

		if (doesFileMatchRegularExpression(file, regular_expression)) {
		    return true;
		}
	    }
	}

	// Try the plugin's default block expression
	if (!default_block_expression.equals("") && doesFileMatchRegularExpression(file, default_block_expression)) {
	    return true;
	}

	// This plugin will (probably) not deal with the specified file
	return false;
    }


    /** Checks whether this plugin instance will process the specified file (given its process_exp). */
    public boolean doesProcessFile(File file)
    {
	// Check the filename against the plugin's process_exp value
	ArrayList arguments = getArguments();
	for (int i = 0; i < arguments.size(); i++) {
	    Argument argument = (Argument) arguments.get(i);
	    if (argument.getName().equals("process_exp")) {
		// Try the assigned value first, for when the user has manually set the value
		String regular_expression = argument.getValue();
		if (regular_expression == null || regular_expression.equals("")) {
		    // Not set, so use the default value
		    regular_expression = argument.getDefaultValue();
		    if (regular_expression.equals("")) {
			continue;
		    }
		}

		if (doesFileMatchRegularExpression(file, regular_expression)) {
		    return true;
		}
	    }
	}

	// Try the plugin's default process expression
	if (!default_process_expression.equals("") && doesFileMatchRegularExpression(file, default_process_expression)) {
	    return true;
	}

	// This plugin will (probably) not deal with the specified file
	return false;
    }


    private boolean doesFileMatchRegularExpression(File file, String regular_expression)
    {
	// The $ at the end doesn't seem to work in Java, so need to add ".*" at the start
	if (regular_expression.startsWith("(?i)")) {
	    // Don't mess up case-insensitive matching though
	    regular_expression = "(?i)" + ".*" + regular_expression.substring("(?i)".length());
	}
	else {
	    regular_expression = ".*" + regular_expression;
	}

	// If the filename matches the regular expression, this plugin will deal with the file in some way
	if (file.getName().matches(regular_expression)) {
	    return true;
	}

	return false;
    }


    public boolean isSeparator() {
	return (element != null && element.getAttribute(StaticStrings.SEPARATOR_ATTRIBUTE).equals(StaticStrings.TRUE_STR));
    }


    public void setDefaultBlockExpression(String default_block_expression)
    {
	this.default_block_expression = default_block_expression;
    }


    public void setDefaultProcessExpression(String default_process_expression)
    {
	this.default_process_expression = default_process_expression;
    }


    public void setDoesExplodeMetadataDatabases(boolean does_explode_metadata_databases)
    {
	this.does_explode_metadata_databases = does_explode_metadata_databases;
    }

    // To work with replace_srcdoc_with_html.pl
    public void setDoesReplaceSrcDocsWithHtml(boolean does_replace_srcdocs_with_html)
    {
	this.does_replace_srcdocs_with_html = does_replace_srcdocs_with_html;
    }

    public void setHasLoadedOptions(boolean hasLoadedOptions)
    {
	this.has_loaded_options = hasLoadedOptions;
    }
}
