/**
 *#########################################################################
 *
 * A component of the Gatherer application, part of the Greenstone digital
 * library suite from the New Zealand Digital Library Project at the
 * University of Waikato, New Zealand.
 *
 * Copyright (C) 2022 New Zealand Digital Library Project
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *########################################################################
 */
package org.greenstone.gatherer.cdm;

import java.util.*;
import org.greenstone.gatherer.DebugStream;
import org.greenstone.gatherer.Gatherer;
import org.greenstone.gatherer.metadata.MetadataElement;
import org.greenstone.gatherer.metadata.MetadataTools;
import org.greenstone.gatherer.util.StaticStrings;
import org.greenstone.gatherer.util.XMLTools;
import org.w3c.dom.*;

/** This class encapsulates a single indexing pair, representing an MG Index
 * it adds level information into the standard index
 * @author John Thompson, Greenstone Digital Library, University of Waikato
 * @version 2.3d
 */
public class MGIndex extends Index {

    /** An values of items in the index level enumeration. */
    static protected final String LEVEL[] = {StaticStrings.DOCUMENT_STR,  StaticStrings.SECTION_STR, StaticStrings.PARAGRAPH_STR};

    /** The level of this index (if old sckool MG). */
    private int level = -1;

    /** Default constructor, which should only be used during DOMProxyListModel creation. */
     public MGIndex() {
     }

    /** Constructor. */
    public MGIndex(Element element) {
    	this.element = element;
    }

    /** Constructor for a newly assigned index with specified level. */
    public MGIndex(int level, ArrayList sources) {
	super(sources);
	this.level = level;
	element.setAttribute(StaticStrings.LEVEL_ATTRIBUTE, LEVEL[level]);
    }

    public MGIndex(String level, ArrayList sources) {
	super(sources);
	for(int i = 0; i < LEVEL.length; i++) {
	    if(LEVEL[i].equalsIgnoreCase(level)) {
		this.level = i;
	    }
	}
	element.setAttribute(StaticStrings.LEVEL_ATTRIBUTE, LEVEL[this.level]);

    }
    
     public DOMProxyListEntry create(Element element) {
     	return new MGIndex(element);
     }
    
    /** Method to get the value of level.
     * @return the level as a int
     */
    public int getLevel() {
	if(level == -1 && element != null) {
	    String level_str = element.getAttribute(StaticStrings.LEVEL_ATTRIBUTE);
	    for(int i = 0; level == -1 && i < LEVEL.length; i++) {
		if(level_str.equals(LEVEL[i])) {
		    level = i;
		}
	    }
	    level_str = null;
	}
	return level;
    }

    public String getID() {
	if(element == null) {
	    id="";
	}
	else if(id == null) {
	    StringBuffer id_buffer = new StringBuffer();
	    // Write level information, if any.
	    int level = getLevel();
	    if(0 <= level && level < 3) {
		id_buffer.append(LEVEL[level]);
		id_buffer.append(StaticStrings.COLON_CHARACTER);
	    }
	    // Write data information. Retrieve each of the content sources and add them in a comma separated list.
	    ArrayList sources = getSources();
	    int sources_size = sources.size();
	    for(int i = 0; i < sources_size; i++) {
		Object source_object = sources.get(i);
		if (source_object instanceof MetadataElement) {
		    String full_element_name = ((MetadataElement)source_object).getFullName();
		    if(full_element_name.startsWith(StaticStrings.EXTRACTED_NAMESPACE) && full_element_name.indexOf(StaticStrings.NS_SEP, StaticStrings.EXTRACTED_NAMESPACE.length()) == -1) {
		       id_buffer.append(full_element_name.substring(StaticStrings.EXTRACTED_NAMESPACE.length()));
		    }
		    else {
			id_buffer.append(full_element_name);
		    }
		}
		else {
		    id_buffer.append(source_object.toString());
		}
		id_buffer.append(StaticStrings.COMMA_CHARACTER);
	    }
	    sources = null;
	    if (id_buffer.length()==0) {
		id = "";
	    } else {
		id = id_buffer.substring(0, id_buffer.length() - 1);
	    }
	}
	return id;
    }


    public void setElement(Element element) {
	this.level = -1;
	super.setElement(element);
    }

    /** Method to set the level of this index which can only be used for the default index. 
     * @param new_level the new level as an int
     */
    public void setLevel(int new_level) {
	// System.err.println("SetLevel(" + new_level + ")");
	if(element != null && element.getNodeName().equals(StaticStrings.INDEX_DEFAULT_ELEMENT)) {
	    if (0 <= new_level && new_level < 3) {
		element.setAttribute(StaticStrings.LEVEL_ATTRIBUTE, LEVEL[new_level]);
	    } else {
		element.setAttribute(StaticStrings.LEVEL_ATTRIBUTE, "");
	    }
	    this.id = null; // Regenerate ID.
	    this.level = new_level;
	}
	else {
	    DebugStream.println("Error! Called setLevel() of index other than the default.");
	}
    }


    /** Method to turn this object into a string representation ready to be placed in the collection configuration file.
     * @return A <strong>String</strong> containing the information of this class.
     */
    public String toString() {
	StringBuffer text_buffer = new StringBuffer("");
	// Generate language dependant id (include extracted metadata namespace)
	// Write level information, if any.
	int level = getLevel();
	if(0 <= level && level < 3) {
	    text_buffer.append(LEVEL[level]);
	    text_buffer.append(StaticStrings.COLON_CHARACTER);
	}
	// Write data information. Retrieve each of the content sources and add them in a comma separated list.
	ArrayList sources = getSources();
	int sources_size = sources.size();
	for(int i = 0; i < sources_size; i++) {
	    String source_name = (sources.get(i)).toString();
	    text_buffer.append(source_name);
	    if(i < sources_size - 1) {
		text_buffer.append(StaticStrings.COMMA_CHARACTER);
	    }
	}
	sources = null;
	return text_buffer.toString();
    }
}
