/**
 *#########################################################################
 *
 * A component of the Gatherer application, part of the Greenstone digital
 * library suite from the New Zealand Digital Library Project at the
 * University of Waikato, New Zealand.
 *
 * <BR><BR>
 *
 * Author: Shaoqun Wu, Greenstone Digital Library, University of Waikato
 *
 * <BR><BR>
 *
 * Copyright (C) 2006 New Zealand Digital Library Project
 *
 * <BR><BR>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * <BR><BR>
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * <BR><BR>
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *########################################################################
 */
package org.greenstone.gatherer.gems;

import javax.swing.table.*; 
import javax.swing.JTable; 
import java.awt.event.ActionListener;
import java.awt.event.ActionEvent;
import java.util.ArrayList; 
import java.awt.Rectangle;
import javax.swing.JOptionPane;
import javax.swing.DefaultCellEditor;
import javax.swing.JTextField;
import javax.swing.ListSelectionModel;

import org.greenstone.gatherer.Dictionary;

public class AttributeTable
    extends JTable 
    implements MetadataElementListener, MetadataSetListener, ActionListener {

    // self reference
    JTable this_table; 
    
    private ArrayList listeners = new ArrayList();
    private JTextField metadata_value_text_field = new JTextField();
    private boolean language_dependent;
    
    public AttributeTable(boolean lang_dependent){
	super();       
        setRowHeight(20);
        this.setComponentOrientation(Dictionary.getOrientation());
        metadata_value_text_field.setComponentOrientation(Dictionary.getOrientation());
	language_dependent = lang_dependent;
	
    	metadata_value_text_field.setBorder(null);
	DefaultCellEditor cellEditor = new DefaultCellEditor(metadata_value_text_field);
	cellEditor.setClickCountToStart(1);
	setDefaultEditor(String.class,cellEditor);   
        this_table = this;
	setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
      }

    /* not currently used */
    public void actionPerformed(ActionEvent e){
	String command = e.getActionCommand();
   
        if (command.equals(GEMSConstants.ADD_ATTRIBUTE)){
	    addAttribute();
 	    return ;
	}
       
       if (command.equals(GEMSConstants.DELETE_ATTRIBUTE)){
           deleteAttribute();
	}   
           
    }

    public void addNewLanguage(String lang) {
	AttributeTableModel model = (AttributeTableModel)getModel();
	model.addNewLanguage(lang);
    }

    public void metadataElementChanged(MetadataElementEvent mee){
	MetadataElementModel model = mee.getMetadataElementModel();      
        if (model == null){ //this model has been deleted
	    setModel(new DefaultTableModel());
            return;
	} 
	populateAttributeTable(model);
    }

    public void metadataSetChanged(MetadataSetEvent mse){
	MetadataSetInfo info = mse.getMetadataSetInfo();
	if (info == null){
	    setModel(new DefaultTableModel());
	    return;
	}
        populateAttributeTable(info);
    }
    
    public void addAttributeListener(AttributeListener al){
	if (!listeners.contains(al)){
	    listeners.add(al);    
	}
    }

    public void removeAttributeListener(AttributeListener al){
	listeners.remove(al);   
    }

    private void addAttribute(){
	AttributeTableModel model = (AttributeTableModel)getModel();
	model.addNewRow();
    }

   
    private void deleteAttribute(){
	int index = getSelectedRow();
	AttributeTableModel model = (AttributeTableModel)getModel();
	model.deleteRow(index);
    }
    
    private void populateAttributeTable(AttributeContainer container){
	AttributeTableModel model = new AttributeTableModel(container, language_dependent);
	setModel(model);
	TableColumnModel tcm = getColumnModel();
	if (language_dependent) {
	    tcm.getColumn(0).setPreferredWidth(100);
	    tcm.getColumn(1).setPreferredWidth(50);
	    tcm.getColumn(2).setPreferredWidth(350);         
	}
	else {
	    tcm.getColumn(0).setPreferredWidth(100);
	    tcm.getColumn(1).setPreferredWidth(400);         
	} 
    }



    private void notifyListeners(Attribute attr){
	AttributeEvent ae = new AttributeEvent(attr);
	for(int i=0;i<listeners.size();i++){
	    AttributeListener al = (AttributeListener) listeners.get(i);
	    al.attributeChanged(ae);
	    
	}
    }



    protected class AttributeTableModel extends AbstractTableModel {
    
	AttributeContainer attribute_container;
	ArrayList attributes;
	boolean language_dependent;
    
	String[] column_names;

	public AttributeTableModel(AttributeContainer container, boolean lang_dependent) {
	    attribute_container = container;
	    language_dependent = lang_dependent;
	    if (language_dependent) {
		attributes = container.getLanguageDependentAttributes();
		column_names = GEMSConstants.LANG_DEPENDENT_ATTRIBUTES_COLUMN_NAMES;
	    } 
	    else {
		attributes = container.getAttributes();
		column_names = GEMSConstants.ATTRIBUTES_COLUMN_NAMES;
	    }
	}

	public int getRowCount(){
	    return attributes.size();
	}
       
	public String getColumnName(int column){
	    return column_names[column];
	}
	
	public Class getColumnClass(int columnIndex){
	    return String.class;
	}    
	public int getColumnCount(){
	    return column_names.length;
	}
	public boolean isCellEditable(int row, int col) {
	    Attribute attr = (Attribute)attributes.get(row);	   
	    if (col == 0 && attr.isRequired()) return false;
	    return true;
	}

	public Object getValueAt(int row, int column){
	    if (row < attributes.size()){
		Attribute attr = (Attribute)attributes.get(row);
		// 	String name = attr.getName();
		//  if (attr.isRequired()){
		//    name = "<html><body Color=blue>"+name+"</body></html>";
		// 	}

		if (column == 0) return attr.getName();
		if (language_dependent) {
		    if (column == 1) return attr.getLanguage();
		    if (column == 2) return attr.getValue();
		}
		else {
		    if (column == 1) return attr.getValue();
		}
	    }	
	    return null; 
	}	    	    

	public void setValueAt(Object value, int row, int column){    
	    if (row < attributes.size() && attributes.size()>0 ){
		Attribute attr = (Attribute)attributes.get(row);
		if (attr.isRequired() && ((String)value).trim().equals("") && !getValueAt(row,column).equals("")) {
		    JOptionPane.showMessageDialog(null,Dictionary.get("GEMS.Attribute_Edition_Error_Message"), Dictionary.get("GEMS.Attribute_Edition_Error"),JOptionPane.ERROR_MESSAGE);
		    return; 
		}
		
		// TODO: check namespace conflict
		if (column == 0) attr.setName((String)value);
		else {
		    if (language_dependent) {
			if (column == 1) attr.setLanguage((String)value);
			else if (column == 2) attr.setValue((String)value); 
		    }
		    else {
			if (column == 1) attr.setValue((String)value);
		    }
		}
		notifyListeners(attr);

		/// check this stuff
		if (attribute_container instanceof MetadataElementModel) {
		    if (attr.getName().trim().equals(GEMSConstants.NAME_ATTRIBUTE)) {
			((MetadataElementModel)attribute_container).setName((String)value);
		    }
		} else if (attribute_container instanceof MetadataSetInfo) {
		    ((MetadataSetInfo)attribute_container).infoChanged(); 
		}
	    }
	}

	/** used only for language dependent attribute tables */
	public void addNewLanguage(String lang)
	{
	    if (!language_dependent) return;
	    int index = attribute_container.languageExists(lang);  
	    if (index >=0){
		this_table.getSelectionModel().setSelectionInterval(index,index);
		Rectangle rect = this_table.getCellRect(index, 0, true);
		this_table.scrollRectToVisible(rect);
	    }
	    else {
		String [] attrNames = attribute_container.getLanguageDependentAttributeNames();
		if (attrNames == null){
		    JOptionPane.showMessageDialog(null,Dictionary.get("GEMS.Add_Lang_Depend_Attr_Error_Message"),Dictionary.get("GEMS.Add_Lang_Depend_Attr_Error"),JOptionPane.ERROR_MESSAGE);
		    return ;
		}
		for(int i=0;i<attrNames.length;i++){
		    Attribute attr = new Attribute(attrNames[i].trim(),"", true);
		    attr.setLanguage(lang);
		    attributes.add(attr); 
		    notifyListeners(attr);
		}
		fireTableDataChanged();
		// first att from new lot
		int first_new_att_index = attributes.size()-attrNames.length;
		this_table.editCellAt(first_new_att_index, 2);
		if (this_table.isEditing()) {
		    this_table.getEditorComponent().requestFocus();
		}
	    
		this_table.getSelectionModel().setSelectionInterval(attributes.size()-attrNames.length,attributes.size()-1); 
		Rectangle rect = this_table.getCellRect(attributes.size()-1, 0, true);
		this_table.scrollRectToVisible(rect); 
	    }
	}
    
	/* not currently used */
	public void addNewRow(){
	    Attribute attr = new Attribute();
	    attributes.add(attr);
	    fireTableDataChanged(); 
	    this_table.getSelectionModel().setSelectionInterval(attributes.size()-1,attributes.size()-1);
	    Rectangle rect = this_table.getCellRect(attributes.size()-1, 0, true);
	    this_table.scrollRectToVisible(rect);    
	}

	/* not currently used */
	public void deleteRow(int index){           
	    if (index < attributes.size() && index >=0){
		Attribute attr = (Attribute)attributes.get(index);          
		if (attr.isRequired()){
		    JOptionPane.showMessageDialog(null,Dictionary.get("GEMS.Attribute_Deletion_Error_Message"), Dictionary.get("GEMS.Attribute_Deletion_Error"),JOptionPane.ERROR_MESSAGE);
		    return;
		}

		// 		int result = JOptionPane.showOptionDialog(null, Dictionary.get("GEMS.Confirm_Removal", Dictionary.get("GEMS.Attribute") + " " +  Dictionary.get("GEMS.Cannot_Undo")), Dictionary.get("GEMS.Confirm_Removal_Title"), JOptionPane.DEFAULT_OPTION, JOptionPane.QUESTION_MESSAGE,null,GEMSConstants.DIALOG_OPTIONS,GEMSConstants.DIALOG_OPTIONS[0] );
         	
		//if (result !=JOptionPane.OK_OPTION) return;
		notifyListeners(attr);
		attributes.remove(index);
		fireTableDataChanged();  
	    }
	
	}
    
    }
}
