/*
 *    TransformingLibrary.java
 *    Copyright (C) 2012 New Zealand Digital Library, http://www.nzdl.org
 *
 *    This program is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
package org.greenstone.gsdl3;

import org.greenstone.gsdl3.util.GSXML;
import org.greenstone.gsdl3.util.XMLTransformer;
import org.greenstone.util.GlobalProperties;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;


import java.io.File;


/**
 * A program to take an GS3 xml file and an XSLT file from the command line and return html
 * @author ak19
 */
final public class TransformingLibrary
{
    // GS3> java -classpath "web/WEB-INF/lib/*":"lib/jni/*":"web/WEB-INF/classes/" org.greenstone.gsdl3.TransformingLibrary localsite default C1.xml CL1.xslt 2>&1 | less

    /**
     *  On Linux, run as:
     *  GS3&gt; java -classpath "web/WEB-INF/lib/*":"lib/jni/*":"web/WEB-INF/classes/" org.greenstone.gsdl3.TransformingLibrary localsite default inputXML inputXSLT
     *  where inputXML is generated by o=xml and inputXSLT is generated by o=skinandlibdoc
     *
     *  Relative paths in xsl:imports made by the input xslt file were solved as described in 
     *  @see http://stackoverflow.com/questions/3699860/resolving-relative-paths-when-loading-xslt-files
     *
     *  For how to include all jars in a folder into the classpath to run a java program, see
     *  @see http://stackoverflow.com/questions/219585/setting-multiple-jars-in-java-classpath
     *  @see http://stackoverflow.com/questions/6780678/run-class-in-jar-file
     */
    public static void main(String args[])
    {
	String usageMsg = "Usage: TransformingLibrary <site name> <interface name> <xml file> <xslt file>";
	
	if (System.getenv("GSDL3SRCHOME") == null) {
	    System.out.println("Before calling this script, run: source gs3-setup.sh\n" + usageMsg);
	    System.exit(1);
	}
	// force GlobalProperties to default GSDL3HOME to GSDL3SRCHOME/web if not already set
	// need GSDL3HOME set for the XMLTransformer.transformer(XSLTfile, XMLfile, interfaceName, DocType)
	GlobalProperties.loadGlobalProperties();
	
	
	if (args.length < 3 || args.length > 4) {
	    System.out.println(usageMsg);
	    System.exit(1);
	}
	
	File xmlFile = new File(args[2]);
	if (!xmlFile.exists()) {
	    System.out.println("File " + args[2] + " does not exist");
	    System.out.println(usageMsg);
	    System.exit(1);
	}
	
	File xsltFile = null;
	if(args.length == 4) {
	    xsltFile = new File(args[3]);
	    if (!xsltFile.exists()) {
		System.out.println("XSLT file " + args[3] + " does not exist");
		System.out.println(usageMsg);
		System.exit(1);
	    }
	}

	
	String siteName = args[0];
	String interfaceName = args[1];
	
	XMLTransformer transformer = new XMLTransformer();
	// Can now call the custom transform() method to do the transformation while 
	// also dealing with relative paths in <xsl:import>s in the input XSLT file
	Node resultNode = transformer.transform(xsltFile, xmlFile, interfaceName, null);
	Element resultElement = resultNode.getOwnerDocument().getDocumentElement();
	System.out.println("*********************");
	System.out.println(GSXML.elementToString(resultElement, true));
	
    }
}
