/**********************************************************************
 *
 * AbstractIsisDb.h
 * Copyright (C) 2003  UNESCO
 *
 * A component of the Greenstone digital library software
 * from the New Zealand Digital Library Project at the
 * University of Waikato, New Zealand.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *********************************************************************/

///////////////////////////////////////////////////////////////////////////////////
// AbstractIsisDb.h -- Abstract Base Class Definitions for an ISIS db
//
//
// The objective is to provide an interface for creating families of Isis
// db without specifying their concrete classes. 
// ABSTRACT FACTORY PATTERN is used.
//
// Author: J-C Dauphin, UNESCO, August 2000
///////////////////////////////////////////////////////////////////////////////////
 
#ifndef __ABSTRACT_ISISDB_H__
#define __ABSTRACT_ISISDB_H__

#include <vector>
#include "IsisDef.h"
#include "mytchar.h"


/////////////////////////////////////////////////////////////////////////////////// 
//   Abstract Base Class, set the DB interface. All db concrete classes must be
//   derived from this class.
//
// Note that this abstract base class can be extended by specifying abstract base
// classes for class class MfFile, class XrfFile, class CFdtCWorksheet, 
// class CDisplayFormat, class CFieldSelectTable, class CSortTable, 
// class MfRecord and struct SMfHeader. Pure virtual methods would then be added
// in the class AbstractIsisDbFactory:
//
// 	virtual AbstractMfFile* makeMfFile() = 0;
//	virtual AbstractXrfFile* makeXrfFile() = 0;
//  ................
 

class MfFile;
class XrfFile;
class CFdt; // Forward 

class CWorksheet;
class CDisplayFormat;
class CFieldSelectTable;
class CSortTable;
class MfRecord;
struct SMfHeader;
 
class AbstractIsisDb 
{
protected:
	ustring    	name;

    MfFile*   mf_;   // Has-a master file
    XrfFile*  xf_;   // Has-a cross ref file
    CFdt*     fdt_;  // Has-a FDT

    std::vector<CWorksheet*>            apWS_;   // Worksheets
    std::vector<CDisplayFormat*>        apDFMT_; // Display formats
    std::vector<CFieldSelectTable*>     apFST_;  // Field Select Tables
    std::vector<CSortTable*>            apST_;   // Sort Tables   
    
public:
	AbstractIsisDb() { };                    // Trivial default C++ constructor
	virtual ~AbstractIsisDb() { };           // Trivial C++ destructor

// Opening the database
   virtual int OpenDb(const TCHAR *fname, FileSystem::AccessMode mode =
	                                          FileSystem::FILE_READWRITE) = 0; 
// Creating the database
   virtual int CreateDb(const TCHAR* fname) = 0;

// Closing the database
   virtual void CloseDb()   = 0;    // Close the db
   virtual void ClearDb()   = 0;    // Clear the db
   virtual void DestroyDb() = 0;    // Delete the db

// Recovery
   virtual bool IsDamaged() = 0;   // Test if the database is damaged
   virtual bool Recover()   = 0;   // Recover the db

// Transactions
   virtual void BeginTransaction()     = 0;   // Begin a transaction
   virtual void CommitTRansaction()    = 0;   // Commit a transaction
   virtual void RollbackTransaction()  = 0;   // Rollback a transaction
   virtual bool InTransaction()        = 0;   // Test if transaction is pending

// (FDT)
   virtual void SetFdt(CFdt *pFdt) { fdt_ = pFdt; }
   virtual CFdt* GetFdt() { return fdt_; }

//  worksheets (WORKSHEETS). 
// (used to create/update the master records of the db)
   virtual void SetWorksheetAt(int i, CWorksheet* pWS) { apWS_.insert(apWS_.begin()+i, pWS); }
   virtual CWorksheet* GetWorksheetAt(int i)           { return apWS_[i]; }
   virtual void RemoveWorksheetAt(int i)               { apWS_.erase(apWS_.begin()+i); }

// display formats (DISPLAY FORMATS).
// (used for specifying the formatting requirements for either on-line
// display of records during searching or for the generation of printed
// output documents.
   virtual void SetDisplayFormatAt(int i, CDisplayFormat* pDFMT) { apDFMT_.insert(apDFMT_.begin()+i, pDFMT); }
   virtual CDisplayFormat* GetDisplayFormatAt(int i)             { return apDFMT_[i]; }
   virtual void RemoveDisplayFormatAt(int i)                     { apDFMT_.erase(apDFMT_.begin()+i); }

// Field Select Table (FST).
// One FST defines the fields of the database to be made searchable
// through the inverted dile.
   virtual void SetFieldSelectTableAt(int i, CFieldSelectTable* pFST) { apFST_.insert(apFST_.begin()+i, pFST); }
   virtual CFieldSelectTable* GetFieldSelectTableAt(int i)            { return apFST_[i]; }
   virtual void RemoveFieldSelectTableAt(int i)                       { apFST_.erase(apFST_.begin()+i); }

   virtual void CreateIndex(int i) = 0;
   virtual void DropIndex(int i)   = 0;

// Sort Select Table (SST).
// One FST defines the fields of the database to be made searchable
// through the inverted dile.

   virtual void SetSortTableAt(int i, CSortTable* pST) { apST_.insert(apST_.begin()+i, pST); }
   virtual CSortTable* GetSortTableAt(int i)           { return apST_[i]; }
   virtual void RemoveSortTableAt(int i)               { apST_.erase(apST_.begin()+i); }

   virtual void  CreateDbRecord(MfRecord &m) = 0;
   virtual void  UpdateDbRecord(MfRecord &m) = 0;
   
   virtual int   ReadDbRecord(mg_s_long mfn, MfRecord &m) = 0;
   virtual int   ReadNextDbRecord(MfRecord &m)       = 0;
   
   virtual bool  DeleteRecord(mg_s_long mfn)              = 0;

   virtual void  PrintDbRecord(MfRecord &m)          = 0;
   virtual void  PrintDbHdr()                        = 0;

   virtual mg_s_long GetNextMfn() = 0;
   virtual mg_s_long GetNextActiveMfn(mg_s_long mfn=-1) = 0;

   virtual SMfHeader& GetSMfHeader() = 0;
   virtual _TCHAR *GetIdxFileName()  = 0;
 

};
 
class AbstractIsisDbFactory
{
public:
	virtual AbstractIsisDb* MakeIsisDb() = 0;
};
#endif   // __ABSTRACT_ISISDB_H__
