/**
 *############################################################################
 * A component of the Greenstone Librarian Interface, part of the Greenstone
 * digital library suite from the New Zealand Digital Library Project at the
 * University of Waikato, New Zealand.
 *
 * Author: Michael Dewsnip, NZDL Project, University of Waikato, NZ
 *
 * Copyright (C) 2005 New Zealand Digital Library Project
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *############################################################################
 */

package org.greenstone.gatherer.util;


import java.awt.*;
import java.io.*;
import java.net.*;
import javax.swing.*;
import org.greenstone.gatherer.DebugStream;


public class JarTools
{
    static private Class root_class = null;
    static private ImageIcon ERROR_ICON = null; 


    static public void initialise(Object root_object)
    {	
	initialise(root_object.getClass());
    }
	
	static public void initialise(Class rootClass)
    {
	root_class = rootClass;
	ERROR_ICON = getImage("error.gif");
    }


    static public void extractFromJar(String filename, String dst_dir, boolean must_be_present) 
    {
	try {
	    // setup input stream for slurping out file
	    InputStream fis = root_class.getResourceAsStream("/"+filename);
	    BufferedInputStream fbis = new BufferedInputStream(fis); 
	    DataInputStream fdbis = new DataInputStream(fbis);

	    // set up output stream for writing to disk
	    String ofname = dst_dir + filename;
	    FileOutputStream fos = new FileOutputStream(ofname);
	    BufferedOutputStream bfos = new BufferedOutputStream(fos);

	    byte[] buf = new byte[1024];
	    int len;
	    int total_bytes = 0;
	    while ((len = fdbis.read(buf)) >= 0) {
		bfos.write(buf,0,len);
		total_bytes += len;
	    }

	    fdbis.close();
	    bfos.close();
	}
	catch (Exception exception) {
	    if (must_be_present) {
		exception.printStackTrace();
	    }
	}
    }


    /** Method to retrieve an image icon with the given filename found in classpath or the resouces directory.
     *  @return The specified <strong>ImageIcon</strong>, or an error image replacement if no such images exists.
     */
    static public ImageIcon getImage(String filename) {
	return getImage(filename, false);
    }

    static public ImageIcon getImage(String filename, boolean wait_until_complete)
    {
	ImageIcon image = null;
	try {
	    image = new ImageIcon(root_class.getResource("/images/" + filename));
	}
	catch (NullPointerException exception) {
	    System.err.println("Error: Could not load image " + filename);
	    DebugStream.println("Error: Could not load image " + filename);
	}

	if (image == null) {
	    image = ERROR_ICON;
	}

	if (wait_until_complete) {
	    int load_status;
	    do {
		load_status = image.getImageLoadStatus();
	    }
	    while (load_status != MediaTracker.ABORTED && load_status != MediaTracker.ERRORED && load_status != MediaTracker.COMPLETE);
	}

	return image;
    }


    static public URL getResource(String resource_name)
    {
	return root_class.getResource(resource_name);
    }


    static public InputStream getResourceAsStream(String resource_name)
    {
	return root_class.getResourceAsStream(resource_name);
    }


    static public boolean isInJar(String filename) 
    {
	InputStream fis = null;
	try {
	    fis = root_class.getResourceAsStream("/" + filename);
		if(fis == null) {
			System.err.println("@@@ JarTools.isInJar(): file inputstream for file /"+filename+" is still null. Not in jar file.");
			return false;
		}
	}
	catch (Exception exception) {
	    exception.printStackTrace();
	    return false;
	}
	finally {
		if(!SafeProcess.closeResource(fis)) {
			System.err.println("@@@ JarTools.isInJar(): Couldn't close file inputstream");
			return false;
		}
	}

	return true;
    }
}
