/**********************************************************************
 *
 * queryinfo.h -- 
 * Copyright (C) 1999  The New Zealand Digital Library Project
 *
 * A component of the Greenstone digital library software
 * from the New Zealand Digital Library Project at the
 * University of Waikato, New Zealand.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *********************************************************************/


#ifndef QUERYINFO_H
#define QUERYINFO_H


#include "gsdlconf.h"
#include "text_t.h"
#include "comtypes.h"

#if defined(GSDL_USE_OBJECTSPACE)
#  include <ospace\std\vector>
#  include <ospace\std\algorithm>
#elif defined(GSDL_USE_STL_H)
#  include <vector.h>
#  if defined(GSDL_USE_ALGO_H)
#    include <algo.h>
#  else
#    include <algorithm.h>
#  endif
#else
#  include <vector>
#  include <algorithm>
#endif

// query parameters

struct queryparamclass {
  text_t combinequery;
  text_t collection;

  // search_index = index+subcollection+language
  text_t index;
  text_t subcollection;
  text_t language;
  text_t level; // for new mg stuff
  text_t querystring;
  int search_type; // 0 = boolean, 1 = ranked
  int match_mode; // 0 = some, 1 = all
  int casefolding;
  int stemming;
  int accentfolding;
  int maxdocs;
  int maxnumeric;
  text_t filterstring; // Filter specified (currently only used by Lucene)
  text_t sortfield; // Field to use for sorting result set (currently used by lucene)
  int sortorder; // 0 = ascending, 1 = descending (only used by Lucene)
  text_t fuzziness; // Search fuzziness amount between 0.0 and 1.0 (only used by Lucene)
  int startresults;
  int endresults;

  queryparamclass ();
  void clear ();
  queryparamclass &operator=(const queryparamclass &q);
};

bool operator==(const queryparamclass &x, const queryparamclass &y);
bool operator!=(const queryparamclass &x, const queryparamclass &y);

// stream output for debugging purposes
ostream &operator<< (ostream &outs, queryparamclass &q);



// term frequencies

struct termfreqclass {
  text_t termstr;
  text_t termstemstr;
  text_tset utf8equivterms; // kept as utf8 string for fast matching
  unsigned int termfreq;

  termfreqclass ();
  void clear();
  termfreqclass &operator=(const termfreqclass &t);
};

typedef vector<termfreqclass> termfreqclassarray;

bool operator==(const termfreqclass &x, const termfreqclass &y);
bool operator!=(const termfreqclass &x, const termfreqclass &y);
bool operator<(const termfreqclass &x, const termfreqclass &y);
bool operator>(const termfreqclass &x, const termfreqclass &y);

// stream output for debugging purposes
ostream &operator<< (ostream &outs, termfreqclass &q);



// one query result

struct docresultclass {
  text_t docid;  // currently used by Lucene only
  int docnum;
  float docweight;
  unsigned int num_query_terms_matched;  // not available on all versions of mg
  int num_phrase_match;                  // not available on all versions of mg

  docresultclass();
  ~docresultclass() {}
  void clear ();
  
  // merges two result classes relating to a single docnum
  docresultclass &combine(const docresultclass &d);

  docresultclass &operator=(const docresultclass &d);
};

bool operator==(const docresultclass &x, const docresultclass &y);
bool operator<(const docresultclass &x, const docresultclass &y);


// stream output for debugging purposes
ostream &operator<< (ostream &outs, docresultclass &a);


struct ltint {
  bool operator()(const int &t1, const int &t2) const
  { return t1 < t2; }
};

typedef map<text_t, docresultclass, lttext_t> docresultmap;



// many document results

struct docresultsclass {
  docresultmap docset;
  vector<text_t> docorder;

  docresultsclass ();
  void clear ();
  void docnum_order();

  void combine_and (const docresultsclass &d);
  void combine_or (const docresultsclass &d);
  void combine_not (const docresultsclass &d);
  
  docresultsclass &operator=(const docresultsclass &d);
};




// query results

struct queryresultsclass {
  queryresultsclass () {clear();}

  text_t error_message;  // Currently only used by Lucene
  int docs_matched; // not available on all versions of mg
  isapprox is_approx;
  //  bool is_approx;   // not available on all versions of mg
  bool syntax_error; // whether the query string was invalid
  bool postprocessed; // whether this record has been post-processed
  
  docresultsclass docs;
  termfreqclassarray orgterms; // terms before they are sorted and uniqued
  termfreqclassarray terms;
  text_tset termvariants;
  text_tset stopwords;

  void clear ();
  queryresultsclass &operator=(const queryresultsclass &q);
  
  void sortuniqqueryterms();
};

// stream output for debugging purposes
ostream &operator<< (ostream &outs, queryresultsclass &q);


#endif
