###########################################################################
#
# DocHistoryFileUtils.pm
#  -- supporting routines for file-level document-version history (fldv-history)
#
# A component of the Greenstone digital library software
# from the New Zealand Digital Library Project at the
# University of Waikato, New Zealand.
#
# Copyright (C) 2023 New Zealand Digital Library Project
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#
###########################################################################

package DocHistoryFileUtils;

# Pragma
use strict;
use warnings;

# use FileHandle;

# Greenstone modules
use dbutil;
use FileUtils;

my $FLDV_HISTORY_DIR = "_fldv_history";

sub prepend_document_version
{
    my ($keepold_doc_dirname,$doc_dirname) = @_;
    
    # Step 5.1:
    #   * Deal with the keepold archives doc's in '_fldv_history' first (if it, and the sub-dirs exist)
    # Step 5.2:
    #   * Deal with the keepold archives doc's top-level content (excluding _fldv_history)

    # For $keepold == 1 this means:
    #   * Recursive hardlink copy each of the keepold 'nminus-<n> folders -> archives as 'nminus-<n+1>'
    #   * Recursive hardlink copy the top-level keepold archive's doc (excluding _fldv_history dir)

    
    my $status_ok = 1;
    
    my $doc_fldv_history_dirname = &FileUtils::filenameConcatenate($doc_dirname,$FLDV_HISTORY_DIR);
    my $keepold_doc_fldv_history_dirname = &FileUtils::filenameConcatenate($keepold_doc_dirname,$FLDV_HISTORY_DIR);

    if (&FileUtils::directoryExists($doc_fldv_history_dirname)) {
	print STDERR "Error - The newly formed 'archives' version of document should not contain the existing '$FLDV_HISTORY_DIR' directory:\n";
	print STDERR "    $doc_fldv_history_dirname\n";
	$status_ok = 0;
    }

    if ($status_ok) {
	# Step 5.1

	# Create _fldv_history in archive doc's dir, ready to take the n -> n+1 copies
	my $mkdir_ok = &FileUtils::makeDirectory($doc_fldv_history_dirname);

	if ($mkdir_ok) {

	    # If the keepold _fldv_history directory exists, then some prior content to deal with.
	    # Otherwise, this is our case of first file-level doc-version history for the doc
	    # and for Step 5.1 there is nothing to do
	    
	    if (&FileUtils::directoryExists($keepold_doc_fldv_history_dirname)) {

		# Start recursive hardlink copying the directories across as n -> n+1

		my $keepold_matching_dirs = &FileUtils::readDirectoryFiltered($keepold_doc_fldv_history_dirname,undef,"^nminus-\\d+\$");

		if (scalar(@$keepold_matching_dirs) == 0) {
		    print STDERR "Warning: no 'nminus-<n>' directories found in $FLDV_HISTORY_DIR:\n";
		    print STDERR "    $keepold_doc_fldv_history_dirname\n";
		    print STDERR "=> No file-level document-version history content to increase by 1\n";
		}
		
		my @sorted_keepold_matching_dirs = sort {
		    my ($a_num) = ($a =~ m/(\d+)$/);
		    my ($b_num) = ($b =~ m/(\d+)$/);
		    
		    # sort into descending order
		    return $b_num <=> $a_num;
		} @$keepold_matching_dirs;
		
		# want sort order to be higest to lowest, for moving 'n' vals up by one
		
		foreach my $nminus_n (@sorted_keepold_matching_dirs) {
		    
		    my $keepold_full_dir = &FileUtils::filenameConcatenate($keepold_doc_fldv_history_dirname,$nminus_n);
		    if (-d $keepold_full_dir) {
			
			my ($n) = ($nminus_n =~ m/(\d+)$/);
			my $new_n = $n + 1;
			
			my $full_new_dir_plus1 = &FileUtils::filenameConcatenate($doc_fldv_history_dirname,"nminus-$new_n");

			my $hard_link_ok = &FileUtils::hardlinkFilesRefRecursive([$keepold_full_dir],$full_new_dir_plus1,
										 { 'strict' => 1 } );

			if (!$hard_link_ok) {
			    print STDERR "Error: Failed to recursive hardlink copy 'nminus-<n>' from 'archives_keepold' to 'archives'\n";
			    $status_ok = 0;
			    last;
			}
			
			#my $hard_link_ok = &FileUtils::hardlinkFilesRefRecursive([$keepold_full_dir],$full_new_dir_plus1,
			#							 { 'strict' => 1, 'exclude_filter_re' => "^$FLDV_HISTORY_DIR\$" } );

			#if (!&FileUtils::renameDirectory($full_dir,$full_new_dir)) {
			#    print STDERR "Error: Failed to shuffled down 'nminus' dirs\n";
			#    $status_ok = 0;
			#    last;
			#}
		    }
		    else {
			print STDERR "Warning: skipping $keepold_full_dir as it is not a directory\n";
		    }
		}
		
	    }
	    else {
		print STDERR "First time file-level document-version history item stored for $doc_fldv_history_dirname\n";
	    }
	}
	else {
	    print STDERR "Error: failed to make directory:\n";
	    print STDERR "    '$doc_fldv_history_dirname'\n";
	    
	    $status_ok = 0;
	}
    }
		

	
	
    # 	# Action Step 5 (from inexport.pm)
    # 	#
    # 	# 5.1 When $keepold == 1
    # 	####   * Move the keepold archives doc's '_fldv_history' to archives (if it exists)
    # 	####   * Then, if nminus-1, nminus-2 etc sub-folders exist, shuffle them (in-situ) down by one

    # 	#   * Recursive hardlink copy each of the keepold 'nminus-<n> folders -> archives as 'nminus-<n+1>'
    # 	#   * Recursive hardlink copy the top-level keepold archive's doc (excluding _fldv_history dir)
	
    # 	# Move the keepold archives doc's '_fldv_history' to archives (if it exists)
    # 	#   * Then, if nminus-1, nminus-2 etc sub-folders exist, shuffle them (in-situ) down by one

	
    # 	if (!&FileUtils::directoryExists($keepold_doc_fldv_history_dirname)) {
    # 	    # First time creating a file-level doc-version history folder for this doc
    # 	    # => make new empty _fldv_history folder in archives dir (nothing else to do at this point)
    # 	    my $mkdir_ok = &FileUtils::makeDirectory($doc_fldv_history_dirname);
	    
    # 	    if (!$mkdir_ok) {
    # 		print STDERR "Error: failed to make directory:\n";
    # 		print STDERR "    '$doc_fldv_history_dirname'\n";
		
    # 		$status_ok = 0;
    # 	    }	    
    # 	}
    # 	else {
    # 	    # Have some existing nminus-n folders to deal this
	    
    # 	    if (&FileUtils::directoryMove($keepold_$doc_fldv_history_dirname,$doc_fldv_history_dirname)) {

    # 		# now shuffle nminus-1, nminus-2 down by one
    # 		my $matching_dirs = &FileUtils::readDirectoryFiltered($doc_fldv_history_dirname,undef,"^nminus-\\d+\$");

    # 		if (scalar(@$matching_dirs) == 0) {
    # 		    print STDERR "Warning: no 'nminus-<n>' directories found in $FLDV_HISTORY_DIR:\n";
    # 		    print STDERR "    $doc_fldv_history_dirname\n";
    # 		    print STDERR "=> No file-level document-version history to store\n";
    # 		}
		
    # 		my @sorted_matching_dirs = sort {
    # 		    my ($a_num) = ($a =~ m/(\d+)$/);
    # 		    my ($b_num) = ($b =~ m/(\d+)$/);
		    
    # 		    # sort into descending order
    # 		    return $b_num <=> $a_num;
    # 		} @$matching_dirs;
		
    # 		# want sort order to be higest to lowest, for moving 'n' vals up by one
		
    # 		foreach my $nminus_n (@sorted_matching_dirs) {
		    
    # 		    my $full_dir = &FileUtils::filenameConcatenate($doc_fldv_history_dirname,$nminus_n);
    # 		    if (-d $full_dir) {
    # 			# print STDERR "    Increasing by one the file-level document-version history number for:\n";
    # 			# print STDERR "      $full_dir\n";
			
    # 			my ($n) = ($nminus_n =~ m/(\d+)$/);
    # 			my $new_n = $n + 1;
			
    # 			my $full_new_dir = &FileUtils::filenameConcatenate($doc_fldv_history_dirname,"nminus-$new_n");
			
    # 			if (!&FileUtils::renameDirectory($full_dir,$full_new_dir)) {
    # 			    print STDERR "Error: Failed to shuffled down 'nminus' dirs\n";
    # 			    $status_ok = 0;
    # 			    last;
    # 			}
    # 		    }
    # 		    else {
    # 			print STDERR "Warning: skipping $full_dir as it is not a directory\n";
    # 		    }
    # 		}
    # 	    }
    # 	    else {
    # 		print STDERR "Error - Failed to move $FLDV_HISTORY_DIR from 'archives_keepold' to 'archives'\n";
    # 		$status_ok = 0;
    # 	    }
    # 	}
    # }
           
    if ($status_ok) {

	# Step 5.2

	#   * Recursive hardlink copy the top-level keepold archive's doc (excluding _fldv_history dir)

	my $doc_fldv_history_dirname_nminus1 = &FileUtils::filenameConcatenate($doc_fldv_history_dirname,"nminus-1");

	my $hard_link_ok = &FileUtils::hardlinkFilesRefRecursive([$keepold_doc_dirname],$doc_fldv_history_dirname_nminus1,
								 { 'strict' => 1, 'exclude_filter_re' => "^$FLDV_HISTORY_DIR\$" } );

	if (!$hard_link_ok) {
	    print STDERR "Error: Failed to recursively hardlink copy the top-level 'keepold_archives' doc as 'archives' $FLDV_HISTORY_DIR/nminus-1 version\n";
	    $status_ok = 0;
	}	
    }

    if (!$status_ok) {
	print STDERR "\n";
	print STDERR "**** A critical error occurred in creating/updating file-level document-version history\n";
	print STDERR "**** After determining and correcting the cause of the error, to reset, delete\n";
	print STDERR "****     your 'archives' folder, and move 'archives_keepold' back to 'archives'\n";
	print STDERR "\n";
	exit 1;
    }
}

sub replace_document_version
{
    my ($keepold_doc_dirname,$doc_dirname) = @_;

    # Step 5.1:
    #   * Deal with the keepold archives doc's in '_fldv_history' first (if it, and the sub-dirs exist)
    # Step 5.2:
    #   * Deal with the keepold archives doc's top-level content (excluding _fldv_history)
    
    # For $replaceold == 1 this means:
    #   * Starting at 'nminus-2', recursive hardlink copy each of the keepold 'nminus-<n>' folders (keeping 'n' the same value)	
    #   * Recursive hardlink copy the top-level keepold archive's doc (excluding _fldv_history dir) as 'nminus-1'
    
    my $status_ok = 1;
    
    my $doc_fldv_history_dirname = &FileUtils::filenameConcatenate($doc_dirname,$FLDV_HISTORY_DIR);
    my $keepold_doc_fldv_history_dirname = &FileUtils::filenameConcatenate($keepold_doc_dirname,$FLDV_HISTORY_DIR);
    
    if (&FileUtils::directoryExists($doc_fldv_history_dirname)) {
	print STDERR "Error - The newly formed 'archives' version of document should not contain the existing '$FLDV_HISTORY_DIR' directory:\n";
	print STDERR "    $doc_fldv_history_dirname\n";
	$status_ok = 0;
    }


    if ($status_ok) {
	# Step 5.1

	# Create _fldv_history in archive doc's dir, ready to take the n -> n copies, starting at n+2 onwards
	my $mkdir_ok = &FileUtils::makeDirectory($doc_fldv_history_dirname);

	if ($mkdir_ok) {

	    # If the keepold _fldv_history directory exists, then some prior content to deal with.
	    # As long as there is a nminus-2, then there are sub-folders to copy
	    # Otherwise there is nothing to do for Step 5.1
	    
	    if (&FileUtils::directoryExists($keepold_doc_fldv_history_dirname)) {

		# Start recursive hardlink copying the directories across n+2 -> n+3 and onwardds

		my $keepold_matching_dirs = &FileUtils::readDirectoryFiltered($keepold_doc_fldv_history_dirname,undef,"^nminus-\\d+\$");

		if (scalar(@$keepold_matching_dirs) == 0) {
		    print STDERR "Warning: no 'nminus-<n>' directories found in $FLDV_HISTORY_DIR:\n";
		    print STDERR "    $keepold_doc_fldv_history_dirname\n";
		    print STDERR "=> No file-level document-version history content to increase by 1\n";
		}
		
		my @sorted_keepold_matching_dirs = sort {
		    my ($a_num) = ($a =~ m/(\d+)$/);
		    my ($b_num) = ($b =~ m/(\d+)$/);
		    
		    # sort into descending order
		    return $b_num <=> $a_num;
		} @$keepold_matching_dirs;
		
		# want sort order to be higest to lowest, for moving 'n' vals up by one
		
		foreach my $nminus_n (@sorted_keepold_matching_dirs) {

		    last if $nminus_n eq "nminus-1";
		    
		    my $keepold_full_dir = &FileUtils::filenameConcatenate($keepold_doc_fldv_history_dirname,$nminus_n);
		    if (-d $keepold_full_dir) {
			
			# my ($n) = ($nminus_n =~ m/(\d+)$/);
			
			my $full_new_dir = &FileUtils::filenameConcatenate($doc_fldv_history_dirname,$nminus_n);

			my $hard_link_ok = &FileUtils::hardlinkFilesRefRecursive([$keepold_full_dir],$full_new_dir,
										 { 'strict' => 1 } );

			if (!$hard_link_ok) {
			    print STDERR "Error: Failed to recursive hardlink copy 'nminus-<n>' from 'archives_keepold' to 'archives'\n";
			    $status_ok = 0;
			    last;
			}
		    }
		    else {
			print STDERR "Warning: skipping $keepold_full_dir as it is not a directory\n";
		    }
		}

	    }
	}
	else {
	    print STDERR "Error: failed to make directory:\n";
	    print STDERR "    '$doc_fldv_history_dirname'\n";
	    
	    $status_ok = 0;
	}
    }

    my $doc_fldv_history_dirname_nminus1 = &FileUtils::filenameConcatenate($doc_fldv_history_dirname,"nminus-1");


    if ($status_ok) {

	# Step 5.2

	#   * Recursive hardlink copy the top-level keepold archive's doc (excluding _fldv_history dir) to archives as nminus-1

	my $doc_fldv_history_dirname_nminus1 = &FileUtils::filenameConcatenate($doc_fldv_history_dirname,"nminus-1");

	my $hard_link_ok = &FileUtils::hardlinkFilesRefRecursive([$keepold_doc_dirname],$doc_fldv_history_dirname_nminus1,
								 { 'strict' => 1, 'exclude_filter_re' => "^$FLDV_HISTORY_DIR\$" } );

	if (!$hard_link_ok) {
	    print STDERR "Error: Failed to recursively hardlink copy the top-level 'keepold_archives' doc as 'archives' $FLDV_HISTORY_DIR/nminus-1 version\n";
	    $status_ok = 0;
	}	
    }

    if (!$status_ok) {
	print STDERR "\n";
	print STDERR "**** A critical error occurred in creating/updating file-level document-version history\n";
	print STDERR "**** After determining and correcting the cause of the error, to reset, delete\n";
	print STDERR "****     your 'archives' folder, and move 'archives_keep' back to 'archives'\n";
	print STDERR "\n";
	exit 1;
    }    
}


sub doc_copy_archivedir_keepold_to_archivedir
{

    my ($keepold_doc_dirname,$doc_dirname) = @_;
    
    my $status_ok = &FileUtils::hardlinkFilesRefRecursive([$keepold_doc_dirname],$doc_dirname, { 'strict' => 1 } );
    
    if (!$status_ok) {
	print STDERR "Error: Failed to recursively hardlink copy the top-level 'keepold_archives' doc to 'archives'\n";
    }	

    if (!$status_ok) {
	print STDERR "\n";
	print STDERR "**** A critical error occurred in creating/updating file-level document-version history\n";
	print STDERR "**** After determining and correcting the cause of the error, to reset, delete\n";
	print STDERR "****     your 'archives' folder, and move 'archives_keep' back to 'archives'\n";
	print STDERR "\n";
	exit 1;
    }        
}


sub archivedir_keepold_to_archivedir
{
    my ($collectcfg, $keepold,$replaceold,$incremental_mode, $archive_info,$archivedir, $archivedir_keepold) = @_;

    # Action Step 5 (from inexport.pm)

    # If needing to turn content in 'archives_keep' into a doc's 'archive' fldv_history/ directory, then
    #   5.1  a keepold doc's '_fldv_history' goes first
    #   5.2  then the keepold doc's top-level content for new 'nminus 1'
    
    my $perform_firsttime_init = 1;
    my $arcinfo_keepold_doc_filename = &dbutil::get_infodb_file_path($collectcfg->{'infodbtype'}, "archiveinf-doc", $archivedir_keepold, $perform_firsttime_init);
    # my $arcinfo_keepold_src_filename = &dbutil::get_infodb_file_path($collectcfg->{'infodbtype'}, "archiveinf-src", $archivedir_keepold, $perform_firsttime_init);
    
    # print STDERR "Creating/Updating archives File-Level Document-Version History (fldv-history) directories:\n";
    my $status_ok = 1;
    
    my $archive_keepold_info = new arcinfo($collectcfg->{'infodbtype'});
    $archive_keepold_info->load_info($arcinfo_keepold_doc_filename);
    
    my $archive_keepold_oidinfo_list = $archive_keepold_info->get_OID_list();
    
    foreach my $keepold_info_array (@$archive_keepold_oidinfo_list) {
	
	my $keepold_OID          = $keepold_info_array->[0];
	my $keepold_doc_file     = $keepold_info_array->[1];
	my $keepold_index_status = $keepold_info_array->[2];
		
	# Work through all entries in arcinfo_keepold doc-id entries:
	#
	#   (1) if archive_info entry is marked for deletion (D)
	#      => Do nothing
	#         This will result in archive_keep and archives versions being wiped
	#         (respectively after import.pl and buildcol.pl
	#
	#   (2) if -keepold on
	#      => move 'archives_keepold' doc's _fldv_history folder to 'archives' version
	#         shuffle all the numbers down by one
	#         move the 'archives_keepold' doc's folder to be the archives _fldv-history/nminus-1
	#
	#   (3) if -replaceold on
	#      => then a similar process to (2), except the 'archives_keepold'
	#         doc folder (without its fldv-history folder), directly replaces the
	#         the 'nminus-1' folder in the 'archives' doc's 'fldv-history' folder
	

	# Indexing cases to consider:
	#   I = needs to be Indexed
	#   R = needs to be Reindexed
	#   D = needs to be Deleted
	#   B = has Been indexed
	#
	# Specifics of what to do for (2) and (3) dependant at times incremental_mode
	print STDERR "Document Versioning: Processing $keepold_OID\n";
	    
	my $oid_info_array = $archive_info->get_info($keepold_OID);
	
	if (defined $oid_info_array) {

	    my $doc_file     = $oid_info_array->[0];
	    my $index_status = $oid_info_array->[1];
	    # For completeness, the other position in oid_info_array are
	    # ->[2] is sort_meta
	    # ->[3] is group_position
	    
	    # **** Need additional work if -groupsize option > 1 is to be supported!
	    # **** otherwise need to have earlier test to prevent -groupsize > 1 from been used with fldv-history

	    my $keepold_doc_filename = &FileUtils::filenameConcatenate($archivedir_keepold, $keepold_doc_file);
	    my $keepold_doc_dirname  = &util::get_parent_folder($keepold_doc_filename);

	    my $doc_filename = &FileUtils::filenameConcatenate($archivedir, $doc_file);
	    my $doc_dirname  = &util::get_parent_folder($doc_filename);

	    if ($index_status eq "D") {
		#
		# (1) if archive_info entry marked for deletion (D)
		#

		# Even though delete is the ultimate outcome for this doc/dir, due to moving 'archives' to 'archives_keep' at the very start of import.pl
		# we need to recursively hardlink copy it back, so the rest of the incremental import.pl process can work as expected
		doc_copy_archivedir_keepold_to_archivedir($keepold_doc_dirname,$doc_dirname);
		
		print STDERR "  The latest version of $keepold_OID is marked for deletion\n";
		print STDERR "  => nothing to do right now (will be deleted when buildcol.pl run)\n";		    
	    }
	    else {
		if ($keepold) {
		    #
		    # (2) if -keepold on
		    #

		    # Still need to consider: I, R, B
		    # R => always trigger store doc-history
		    # I => always trigger store doc-history
		    # B => trigger store doc-history if incremental_mode != "all"
		    
		    if ($index_status eq "R") {
			print STDERR "  Updated version of document from import directory (Index-Status=R)\n";
			print STDERR "  => keepold: storing snapshot of previous version in $FLDV_HISTORY_DIR as 'nminus-1'\n";
			prepend_document_version($keepold_doc_dirname,$doc_dirname);
		    }
		    elsif ($index_status eq "I") {

			if (!&FileUtils::directoryExists($doc_dirname)) {
			    # This can happen if import.pl has been run twice in a row , without buildcol.pl
			    #
			    # Specifics: In the case that the second import.pl run is incremental, then previous "I" entries
			    # from the first import.pl have not been processed by buildcol.pl and turned into "B" entries.
			    # However, if existing "I" entries have not been modified between the first and second import.pl
			    # then there is is no 'doc_dirname' that has been created in the newly formed archives.  Further,
			    # the archives_keepold version isn't an example of a document that needs a "fldv" bump.

			    # long description to say, that basically we want to treat the situation like a "B", that
			    # it's been index

			    print STDERR "  Unchanged version of document in 'archives/' however (Index-Status=I not B)\n";
			    print STDERR "  This can occur if two consecutive import.pl's have been run, without a buildcol.pl inbetween\n";
			    print STDERR "  => Recusirvely copying the archives_keepold/ doc across verbatim to archives/\n";
			    print STDERR "     (effectively treating as a 'B')\n";
				
			    doc_copy_archivedir_keepold_to_archivedir($keepold_doc_dirname,$doc_dirname);			    
			}
			else {
			    print STDERR "  Newly generated version of document from 'import/' of existing document from 'archives/' (Index-Status=I)\n";
			    print STDERR "  => keepold: store snapshot of previous version in $FLDV_HISTORY_DIR as 'nminus-1'\n";
			    
			    prepend_document_version($keepold_doc_dirname,$doc_dirname);
			}
		    }
		    elsif ($index_status eq "B") {
			# This doc's archive doc folder hasn't change, but due to moving 'archives' to 'archives_keep' at the very start of import.pl
			# we need to recursively hardlink copy it back
			
			doc_copy_archivedir_keepold_to_archivedir($keepold_doc_dirname,$doc_dirname);
			
			if ($incremental_mode ne "all") {
			    print STDERR "  Unchanged version of document in 'archives/' (Index-Status=B)\n";
			    print STDERR "  => keepold without incremental: store snapshot of previous version in $FLDV_HISTORY_DIR as 'nminus-1'\n";
			    prepend_document_version($keepold_doc_dirname,$doc_dirname);			    
			}
			else {
			    print STDERR "  Unchanged version of document in 'archives/' (Index-Status=B)\n";
			    print STDERR "  => no snapshot store needed when in incremental mode\n";
			}
		    }
		}
		elsif ($replaceold) {
		    #
		    # (3) if -replaceold on
		    #

		    # Still need to consider: I, R, B
		    # R => always trigger replace nminus-1 doc-history
		    # I => always trigger replace nminus-1 doc-history
		    # B => trigger replace nminus-1 doc-history if incremental_mode != "all"
		    
		    if ($index_status eq "R") {
			print STDERR "  Updated version of document from import directory (Index-Status=R)\n";
			print STDERR "  => replaceold: replacing snapshot of previous version in $FLDV_HISTORY_DIR as 'nminus-1'\n";
			replace_document_version($keepold_doc_dirname,$doc_dirname);
		    }
		    elsif ($index_status eq "I") {

			if (!&FileUtils::directoryExists($doc_dirname)) {
			    # This can happen if import.pl has been run twice in a row , without buildcol.pl
			    #
			    # For explanation see above 'I' entry for -keepold
			    #
			    # Basically we want to treat the situation like a "B", that it's been index

			    print STDERR "  Unchanged version of document in 'archives/' however (Index-Status=I not B)\n";
			    print STDERR "  This can occur if two consecutive import.pl's have been run, without a buildcol.pl inbetween\n";
			    print STDERR "  => Recusirvely copying the archives_keepold/ doc across verbatim to archives/\n";
			    print STDERR "     (effectively treating as a 'B')\n";
				
			    doc_copy_archivedir_keepold_to_archivedir($keepold_doc_dirname,$doc_dirname);
			}
			else {
			    print STDERR "  Newly generated version of document from 'import/' of existing document from 'archives/' (Index-Status=I)\n";
			    print STDERR "  => keepold: store snapshot of previous version in $FLDV_HISTORY_DIR as 'nminus-1'\n";
			    replace_document_version($keepold_doc_dirname,$doc_dirname);
			}
		    }
		    elsif ($index_status eq "B") {
			if ($incremental_mode ne "all") {
			    print STDERR "  Unchanged version of document in 'archives/' (Index-Status=B)\n";
			    print STDERR "  => keepold without incremental: store snapshot of previous version in $FLDV_HISTORY_DIR as 'nminus-1'\n";
			    replace_document_version($keepold_doc_dirname,$doc_dirname);			    
			}
			else {
			    print STDERR "  Unchanged version of document in 'archives/' (Index-Status=B)\n";
			    print STDERR "  => no snapshot store needed when in incremental mode\n";
			}
		    }

		    
		}
		else {
		    print STDERR "**** Error: Encountered a condition for $keepold_OID that was not expected\n";
		}
	    }
	    
	}
	else {
	    print STDERR "**** Error: Did not find 'archives/' matching document for 'archives_keepold/' OID=$keepold_OID\n";
	    print STDERR "Skipping!\n";
	}
    }

    if ($status_ok) {
	&FileUtils::removeFilesRecursive($archivedir_keepold);
    }
}

1;
