/**
 *#########################################################################
 *
 * A component of the Gatherer application, part of the Greenstone digital
 * library suite from the New Zealand Digital Library Project at the
 * University of Waikato, New Zealand.
 *
 * <BR><BR>
 *
 * Author: John Thompson, Greenstone Digital Library, University of Waikato
 *
 * <BR><BR>
 *
 * Copyright (C) 1999 New Zealand Digital Library Project
 *
 * <BR><BR>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * <BR><BR>
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * <BR><BR>
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *########################################################################
 */
package org.greenstone.gatherer.gui;

import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.event.*;
import org.greenstone.gatherer.Configuration;
import org.greenstone.gatherer.Dictionary;
import org.greenstone.gatherer.Gatherer;
import org.greenstone.gatherer.util.JarTools;
import org.greenstone.gatherer.util.Utility;

/** The menu bar for the Gatherer main GUI. 
 * @author John Thompson, Greenstone Digital Library, University of Waikato
 * @version 2.2
 */
public class MenuBar
    extends JMenuBar
{
    static public ImageIcon BLANK_ICON = JarTools.getImage("blank.gif");
    static public ImageIcon HELP_ICON = JarTools.getImage("help.png");

    /** The icon to be displayed alongside the context choosen help file. */
    private int current_tab          = -1;
    private int current_mode;

    public JMenu file               = null;
    public JMenu edit               = null;
    public JMenu help                = null;
    public JMenuItem file_associations;
    public JMenuItem file_cdimage    = null;
    public JMenuItem file_close      = null;
    public JMenuItem file_delete     = null;
    public JMenuItem file_exit       = null;
    public JMenuItem file_exportas   = null;
    public JMenuItem file_exportmeta = null;
    public JMenuItem file_convertmeta   = null;
    public JMenuItem file_new        = null;
    public JMenuItem file_open       = null;
    public JMenuItem file_options    = null;
    public JMenuItem file_save       = null;
    public JMenuItem edit_copy;
    public JMenuItem edit_cut;
    public JMenuItem edit_paste;
    public JMenuItem edit_config;
    public JMenuItem edit_group_config;
    public JMenuItem help_general;
    public JMenuItem help_download;
    public JMenuItem help_gather;
    public JMenuItem help_enrich;
    public JMenuItem help_design;
    public JMenuItem help_create;
    public JMenuItem help_format;
  public JMenuItem help_files;
    public JMenuItem help_about;


    public MenuBar(MenuListener menu_listener)
    {
        this.setComponentOrientation(Dictionary.getOrientation());
        current_mode = Configuration.getMode();

	file = new JMenu();
	file.setEnabled(false);
	file.setText(Dictionary.get("Menu.File"));
	file.setComponentOrientation(Dictionary.getOrientation());
        
	file_associations = new JMenuItem(Dictionary.get("Menu.File_Associations"));
	file_associations.addActionListener(Gatherer.g_man);	  
	file_associations.setComponentOrientation(Dictionary.getOrientation());
        
	file_cdimage = new JMenuItem(Dictionary.get("Menu.File_CDimage"));
	file_cdimage.addActionListener(Gatherer.g_man);
	file_cdimage.setEnabled(!Gatherer.isGsdlRemote);
	file_cdimage.setComponentOrientation(Dictionary.getOrientation());
        
	file_close = new JMenuItem(Dictionary.get("Menu.File_Close"));
	file_close.addActionListener(Gatherer.g_man);
	file_close.setEnabled(false);
	file_close.setComponentOrientation(Dictionary.getOrientation());
        
	file_delete = new JMenuItem(Dictionary.get("Menu.File_Delete"));
	file_delete.addActionListener(Gatherer.g_man);
	file_delete.setComponentOrientation(Dictionary.getOrientation());
        
	file_exit = new JMenuItem(Dictionary.get("Menu.File_Exit"));
	file_exit.addActionListener(Gatherer.g_man);
	file_exit.setComponentOrientation(Dictionary.getOrientation());
        
	file_exportas = new JMenuItem(Dictionary.get("Menu.File_ExportAs"));
	file_exportas.addActionListener(Gatherer.g_man);
        // later, allow this for webswing
	file_exportas.setEnabled(!(Gatherer.isGsdlRemote || Gatherer.isWebswing));
	file_exportas.setComponentOrientation(Dictionary.getOrientation());

	file_exportmeta = new JMenuItem(Dictionary.get("Menu.File_ExportMeta"));
	file_exportmeta.addActionListener(Gatherer.g_man);
	file_exportmeta.setEnabled(false);
	file_exportmeta.setComponentOrientation(Dictionary.getOrientation());

	file_convertmeta = new JMenuItem(Dictionary.get("Menu.File_ConvertMetaToCSV"));
	file_convertmeta.addActionListener(Gatherer.g_man);
	file_convertmeta.setEnabled(false);
	file_convertmeta.setComponentOrientation(Dictionary.getOrientation());
	
	file_new = new JMenuItem(Dictionary.get("Menu.File_New"));
	file_new.addActionListener(Gatherer.g_man);
	file_new.setComponentOrientation(Dictionary.getOrientation());
        
	file_open = new JMenuItem(Dictionary.get("Menu.File_Open"));
	file_open.addActionListener(Gatherer.g_man);
	file_open.setComponentOrientation(Dictionary.getOrientation());
        
	file_options = new JMenuItem(Dictionary.get("Menu.File_Options"));
	file_options.addActionListener(Gatherer.g_man);
	file_options.setComponentOrientation(Dictionary.getOrientation());
	
	file_save = new JMenuItem(Dictionary.get("Menu.File_Save"));
	file_save.addActionListener(Gatherer.g_man);
	file_save.setEnabled(false);
	file_save.setComponentOrientation(Dictionary.getOrientation());
        
	// Layout (file menu)
	file.add(file_new);
	file.add(file_open);
	file.add(file_save);
	file.add(file_close);
	file.add(new JSeparator());
	file.add(file_delete);
	file.add(file_exportas);
	file.add(file_exportmeta);
	file.add(file_convertmeta);
	// these currently don't work. better to just disable them?
	if (!Gatherer.GS3) {
	    file.add(file_cdimage);
	}
	file.add(new JSeparator());
	file.add(file_associations);
	file.add(file_options);
	file.add(new JSeparator());
	file.add(file_exit);

        // Disable File->New if we are webswing and user has no creation permissions
        if (Gatherer.isWebswing && !Gatherer.webswingAuthenticator.canCreateNewCollections()) {
          file_new.setEnabled(false);
        }
	// Edit menu
	edit = new JMenu();
	edit.setEnabled(false);
	edit.setText(Dictionary.get("Menu.Edit"));
	edit.setComponentOrientation(Dictionary.getOrientation());
	String modkey = "ctrl";
	if(Utility.isMac()) { // on Mac, we now use the Apple key instead of Ctrl for GLI/GEMS edit shortcuts
	    // http://stackoverflow.com/questions/5585919/creating-unicode-character-from-its-number
	    char appleKeyCodepoint = 0x2318; // applekey symbol unicode codepoint: U+2318 (\u2318)
	    String appleKeySymbol = String.valueOf(appleKeyCodepoint);
	    modkey = appleKeySymbol;
	}

	edit_cut = new JMenuItem(Dictionary.get("Menu.Edit_Cut", modkey));
	edit_cut.addActionListener(Gatherer.g_man);
	edit_cut.setComponentOrientation(Dictionary.getOrientation());
        
	edit_copy = new JMenuItem(Dictionary.get("Menu.Edit_Copy", modkey));
	edit_copy.addActionListener(Gatherer.g_man);
        edit_copy.setComponentOrientation(Dictionary.getOrientation());
	
	edit_paste = new JMenuItem(Dictionary.get("Menu.Edit_Paste", modkey));
	edit_paste.addActionListener(Gatherer.g_man);
	edit_paste.setComponentOrientation(Dictionary.getOrientation());

	// Layout (edit menu)
	edit.add(edit_cut);
	edit.add(edit_copy);
	edit.add(edit_paste);

	// if GS3, then we have a menu item that allows editing of config files
	if(Gatherer.GS3) {   

	    edit_config = new JMenuItem(Dictionary.get("Menu.Edit_Config"));
	    // handle the actual Edit > ColConfig.xml menu item
	    edit_config.addActionListener(Gatherer.g_man);
	    edit_config.setComponentOrientation(Dictionary.getOrientation());
	    
	    // The Edit menu itself now listens, in order to gray out the
	    // Edit > collectionConfig.xml option when no collection is open
	    // (JMenu doesn't work with ActionListener only with MenuListener, see
	    // http://stackoverflow.com/questions/9862165/jmenu-actionlistener)
	    edit.addMenuListener(menu_listener);

 	    edit_group_config = new JMenuItem(Dictionary.get("Menu.Edit_Group_Config"));
	    // handle the actual Edit > groupConfig.xml menu item
	    edit_group_config.addActionListener(Gatherer.g_man);
	    edit_group_config.setComponentOrientation(Dictionary.getOrientation());
            if (current_mode == Configuration.EXPERT_MODE) {
              // only display this option if in expert mode
              edit.add(edit_config);
              if (Gatherer.isWebswing) {
                // for now, if you can edit all colls, or edit shared colls,
                // you are allowed to edit group config
                if (Gatherer.webswingAuthenticator.canEditSharedCollections() ) {
                  edit.add(edit_group_config);
                }
              } else {
                edit.add(edit_group_config);
              }
            }

	}
    
	// Help menu
	help = new JMenu();
	help.setIcon(HELP_ICON);
	help.setText(Dictionary.get("Menu.Help"));
	help.setComponentOrientation(Dictionary.getOrientation());
        
	help_general = new JMenuItem(Dictionary.get("Source.General"));
	help_general.addActionListener(Gatherer.g_man);
	help_general.setComponentOrientation(Dictionary.getOrientation());
        
	help_download = new JMenuItem(Dictionary.get("GUI.Download"), BLANK_ICON);
	help_download.addActionListener(Gatherer.g_man);
	help_download.setComponentOrientation(Dictionary.getOrientation());
        
	help_gather = new JMenuItem(Dictionary.get("GUI.Gather"), BLANK_ICON);
	help_gather.addActionListener(Gatherer.g_man);
	help_gather.setComponentOrientation(Dictionary.getOrientation());
        
	help_enrich = new JMenuItem(Dictionary.get("GUI.Enrich"), BLANK_ICON);
	help_enrich.addActionListener(Gatherer.g_man);
	help_enrich.setComponentOrientation(Dictionary.getOrientation());
        
	help_design = new JMenuItem(Dictionary.get("GUI.Design"), BLANK_ICON);
	help_design.addActionListener(Gatherer.g_man);
	help_design.setComponentOrientation(Dictionary.getOrientation());
        
	help_create = new JMenuItem(Dictionary.get("GUI.Create"), BLANK_ICON);
	help_create.addActionListener(Gatherer.g_man);
	help_create.setComponentOrientation(Dictionary.getOrientation());
        
	help_format = new JMenuItem(Dictionary.get("GUI.Format"), BLANK_ICON);
	help_format.addActionListener(Gatherer.g_man);
	help_format.setComponentOrientation(Dictionary.getOrientation());
        
	help_files = new JMenuItem(Dictionary.get("GUI.Files"), BLANK_ICON);
	help_files.addActionListener(Gatherer.g_man);
	help_files.setComponentOrientation(Dictionary.getOrientation());
        
	help_about = new JMenuItem(Dictionary.get("Menu.Help_About"));
	help_about.addActionListener(Gatherer.g_man);
	help_about.setComponentOrientation(Dictionary.getOrientation());
        
	// Layout (help menu)
	help.add(help_general);
	help.add(new JSeparator());
	if (Configuration.get("workflow.download", true) && Gatherer.isDownloadEnabled) {
	    help.add(help_download);
	}
	if (Configuration.get("workflow.gather", true)) {
	    help.add(help_gather);
	}
	if (Configuration.get("workflow.enrich", true)) {
	    help.add(help_enrich);
	}
	if (Configuration.get("workflow.design", true)) {
	    help.add(help_design);
	}
	if (Configuration.get("workflow.create", true)) {
	    help.add(help_create);
	}
	if (Configuration.get("workflow.format", true)) {
	    help.add(help_format);
	}
	if (Configuration.get("workflow.files", true)) {
	    help.add(help_files);
	}
	help.add(new JSeparator());
	help.add(help_about);

	// Layout (menu bar)
	this.add(file);
	this.add(Box.createHorizontalStrut(15));
	this.add(edit);
	this.add(Box.createHorizontalGlue());
	this.add(help);
    }

    /** Once a quit has been requested by the user, prevent any further menu selections. */
    public void exit() {
	file.setEnabled(false);
	edit.setEnabled(false);
	help.setEnabled(false);
    }

    public void refresh(int refresh_reason, boolean ready)
    {
	file_close.setEnabled(ready);
	file_save.setEnabled(ready);
        file_exportmeta.setEnabled(ready);
        file_convertmeta.setEnabled(ready);
    }


    /** In order to provide context aware help advice we keep track of which
     *  tab the user has open, and then highlight that help menu item with
     *  separators.
     *  @param tab_index The index of the selected tab (0-7).
     */
    public void tabSelected(int tab_index) {
	JMenuItem selected;
	if(current_tab != -1) {
	    // Remove the image
	    selected = help.getItem(current_tab);
	    if(selected != null) {
		selected.setIcon(BLANK_ICON);
	    }
	}
	current_tab = tab_index + 2;
	selected = help.getItem(current_tab);
	if(selected != null) {
	    selected.setIcon(HELP_ICON);
	}
	selected = null;
    }

  public void modeChanged(int mode) {

    if (mode == Configuration.EXPERT_MODE) {
      // add edit config files
      edit.add(edit_config);
      if (Gatherer.isWebswing) {
        if (Gatherer.webswingAuthenticator.canEditSharedCollections()) {
          edit.add(edit_group_config);
        }
      } else {
        edit.add(edit_group_config);
      }
    }
    else {
      edit.remove(edit_config);
      edit.remove(edit_group_config);
    }
  }
}
