#!/usr/bin/env perl
use warnings;
use strict;

###########################################################################
#
# pdfpstoimg.pl -- convert PDF or PS documents to various types of Image format
#
# A component of the Greenstone digital library software
# from the New Zealand Digital Library Project at the 
# University of Waikato, New Zealand.
#
# Copyright (C) 2001 New Zealand Digital Library Project
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#
###########################################################################
# pdfpstoimg.pl is a wrapper for running the ImageMagick 'convert' utility 
# which converts PDF and PS documents to various types of image (e.g. PNG, 
# GIF, JPEG format). We then create an item file to join the images together
# into a document. The item file will be processed by PagedImagePlugin

BEGIN {
    die "GSDLHOME not set\n" unless defined $ENV{'GSDLHOME'};
    unshift (@INC, "$ENV{'GSDLHOME'}/perllib");
}

use parsargv;
use util;
#use FileUtils;
use Cwd;
use File::Basename;

sub print_usage {
    print STDERR  
	("images2item.pl script for generating an item file that corresponds to a series of images.\n",
	 "Usage: images2item.pl [options] <image1> [<opt-txt1>] <image2> [<opt-txt2] ... <imageN> <[opt-txtN]> <output-file.item>\n",
	 "Options:\n",
	 "\t-convert_to\toutput type (pagedimg,pagedimgtxt)\n"
	 
	);
    exit (1);
}

sub main {
    my (@ARGV) = @_;
    my ($convert_to, $autogen_item_filename);
    
    if (!parsargv::parse(\@ARGV,
    			 'convert_to/(pagedimg)|(pageimgtxt)/pagedimgtxt', \$convert_to,
			 'autogen_item_filename', \$autogen_item_filename,
    			 )) {
    	print_usage();
    }
    
    # Make sure the user has specified both input and output files
    my $item_file = undef;

    if ($autogen_item_filename) {
	if (scalar(@ARGV)<1) {
	    print_usage();
	}
    }
    else {
	$item_file = pop @ARGV;

	if ($item_file =~ m/\.(gif|png|jpg|jpeg|tif|tiff|jp2)$/i) {
	    print STDERR "Error: output item file cannot use an image file extension\n";
	    print_usage();
	}
    
	if ($item_file !~ m/\.item$/i) {
	    print STDERR "Warning: output item file '$item_file' uses non-standard file-extension\n";
	}	
    }

    # Treat everything else as input files
    my @input_images_and_opttxt_files = @ARGV;

    my $starts_at_0 = 0;
    # To generate page title metadata in .item file, need to determine if first page number starts at 0 or not
    # To be compatible with older util::create_item approach, skip over any .item files that have made it into
    # the list of files we are processing
    while ((scalar(@input_images_and_opttxt_files)>0) && ($input_images_and_opttxt_files[0] =~ m/\.item$/i)) {
	shift @input_images_and_opttxt_files;
    }

    if (scalar(@input_images_and_opttxt_files)>0) {
	my $firstfile = $input_images_and_opttxt_files[0];
    
	if(&util::image_page_number($firstfile) == 0) { # 00 will evaluate to 0 too in this condition
	    $starts_at_0 = 1;
	}
    }
    else {
	print STDERR "After ignoring .item files on the command-line arguments, no image or text filenames specified\n";
	print_usage();
    }
    
    if ($autogen_item_filename) {
	my $shared_file_prefix = &util::longest_common_prefix(\@input_images_and_opttxt_files);
	$shared_file_prefix =~ s/[-\.]*$//;
	
	$item_file = "$shared_file_prefix.item";	    
    }
    
    my $item_filename = Cwd::abs_path($item_file);
    my $output_dir = &File::Basename::dirname($item_filename);

    print "Outputting item file: $item_file\n";
    &util::create_itemfile_from_list(\@input_images_and_opttxt_files,$convert_to,$starts_at_0, $output_dir,$item_filename);

    return 0;
}

# indicate our error status, 0 = success
exit (&main(@ARGV));



