/**
 *#########################################################################
 *
 * A component of the Gatherer application, part of the Greenstone digital
 * library suite from the New Zealand Digital Library Project at the
 * University of Waikato, New Zealand.
 *
 * <BR><BR>
 *
 * Author: John Thompson, NZDL Project, University of Waikato
 *
 * <BR><BR>
 *
 * Copyright (C) 2005 New Zealand Digital Library Project
 *
 * <BR><BR>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * <BR><BR>
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * <BR><BR>
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *########################################################################
 */
package org.greenstone.gatherer.file;

import java.io.File;
import javax.swing.*;

import org.webswing.toolkit.api.WebswingUtil;

import org.greenstone.gatherer.Configuration;
import org.greenstone.gatherer.Dictionary;
import org.greenstone.gatherer.Gatherer;
import org.greenstone.gatherer.collection.CollectionManager;
import org.greenstone.gatherer.collection.CollectionTree;
import org.greenstone.gatherer.collection.CollectionTreeNode;
import org.greenstone.gatherer.collection.FullCollectionTreeNode;
import org.greenstone.gatherer.gui.ExplodeMetadataDatabasePrompt;
import org.greenstone.gatherer.gui.ReplaceSrcDocWithHtmlPrompt; 
import org.greenstone.gatherer.gui.GProgressBar;
import org.greenstone.gatherer.gui.NewFolderOrFilePrompt;
import org.greenstone.gatherer.gui.RenamePrompt;
import org.greenstone.gatherer.gui.tree.DragTree;
import org.greenstone.gatherer.remote.RemoteGreenstoneServer;
import org.greenstone.gatherer.util.DragComponent;
import org.greenstone.gatherer.util.Utility;
import org.greenstone.gatherer.DebugStream;

/** Manages the moving of files within a separate thread.
 * @author John Thompson, NZDL Project, University of Waikato
 */
public class FileManager
{
    /** Not only the queue of files to be moved, but also the object that moves them. */
    static private FileQueue file_queue = null;

    public static final int COPY = 0;
    public static final int MOVE = 1;

    public static final int FILE_TYPE = 0;
    public static final int DUMMY_TYPE = 1;
    public static final int FOLDER_TYPE = 2;

    protected static File startup_directory = null;


    /** Constructor.
     * @see org.greenstone.gatherer.file.FileQueue
     */
    public FileManager()
    {
	file_queue = new FileQueue();
	file_queue.start();
    }


    /** Determine what action should be carried out by the file queue, and add all of the necessary file jobs. */
    public void action(DragComponent source, FileNode[] source_nodes, DragComponent target, FileNode target_node)
    {
	// Check there is something to do
	if (source_nodes == null || source_nodes.length == 0) {
	    return;
	}

	// We need a unique ID for each file task
	long id = System.currentTimeMillis();

	// If source and target are the same we're moving
	if (source == target) {
	    // Start a new move FileTask and we're done
	    (new FileTask(id, source, source_nodes, target, target_node, FileJob.MOVE)).start();
	    return;
	}

	// If target isn't the RecycleBin, we're copying
	if (!(target instanceof RecycleBin)) {
	    // Start a new copy FileTask and we're done
	    (new FileTask(id, source, source_nodes, target, target_node, FileJob.COPY)).start();
	    return;
	}

	// We're deleting... but first make sure source isn't read-only
	boolean read_only_source = false;

	// The workspace tree is read-only...
	if (source.toString().equals("Workspace")) {
	    read_only_source = true;

	    // ...except for files from the "Downloaded Files" folder
	    String downloaded_files_folder_path = Gatherer.getGLIUserCacheDirectoryPath();
	    for (int i = 0; i < source_nodes.length; i++) {
		// Is this the "Downloaded Files" folder?
		if (source_nodes[i].getFile().getAbsolutePath().startsWith(downloaded_files_folder_path)) {
		    read_only_source = false;
		} else {
                  // at least one file is not in downloaded area
                  read_only_source = true;
                  break;
                }
	    }
	}

	// The source is read-only, so tell the user and abort
	if (read_only_source) {
	    JOptionPane.showMessageDialog(Gatherer.g_man, Dictionary.get("FileActions.Read_Only"), Dictionary.get("General.Error"), JOptionPane.ERROR_MESSAGE);
	    return;
	}

	// Start a new delete FileTask and we're done
	(new FileTask(id, source, source_nodes, target, target_node, FileJob.DELETE)).start();
    }

    /** For moving and copying of folders. */
    public void action(File sourceFolder, File targetFolder, int operation) {
	(new SimpleFileTask(sourceFolder, targetFolder, operation)).start();
    }
    

    /** Retrieves the file queue object. */
    public FileQueue getQueue()
    {
	return file_queue;
    }

    /** Performs the simple file task of moving or copying folders. */
    private class SimpleFileTask
	extends Thread
    {
	private File sourceFolder;
	private File targetFolder;
	int operation; // MOVE or COPY

	public SimpleFileTask(File sourceFolder, File targetFolder, int operation)
	{
	    this.sourceFolder = sourceFolder;	    
	    this.targetFolder = targetFolder;
	    this.operation = operation;
	}


	public void run()
	{
	    // check if we're moving or overwriting the current collection
	    String currentColPath = Gatherer.getCollectDirectoryPath()+CollectionManager.getLoadedCollectionName();
	    if(currentColPath.equals(sourceFolder.getAbsolutePath()) 
	       || currentColPath.equals(targetFolder.getAbsolutePath())) {
		Gatherer.g_man.saveThenCloseCurrentCollection();
	    }
	    
	    // if moving, try a simple move operation (if it works, it
	    // shouldn't take long at all and doesn't need a progress bar)
	    if(operation == MOVE && sourceFolder.renameTo(targetFolder)) {
		//System.err.println("**** A simple renameTo() worked.");
		//WorkspaceTreeModel.refreshGreenstoneCollectionsNode();
              Gatherer.g_man.refreshWorkspaceTreeGreenstoneCollections();
		return;
	    }

	    // Reset the progress bar and set it to indeterminate while calculating its size
	    GProgressBar progress_bar = file_queue.getProgressBar();
	    progress_bar.reset();
	    progress_bar.setIndeterminate(true);

	    String status = "FileActions.Moving";
	    if(operation == COPY) {
		status = "FileActions.Copying";	    
	    }
	    progress_bar.setString(Dictionary.get(status));
	    file_queue.getFileStatus().setText(Dictionary.get(status,
							      file_queue.formatPath(status,
									 sourceFolder.getAbsolutePath(), 
									 file_queue.getFileStatus().getSize().width)));

	    // do the move or copy operation
	    try {
		//System.err.println("**** Copying " + sourceFolder + " to: " + targetFolder);
		file_queue.copyDirectoryContents(sourceFolder, targetFolder);		
	    } catch(Exception e) {
		JOptionPane.showMessageDialog(Gatherer.g_man, e.getMessage(), 
					      "Can't perform file operation", JOptionPane.ERROR_MESSAGE);

		progress_bar.setIndeterminate(false);
		progress_bar.clear();
		return;
	    }
		
	    // if moving, delete the original source folder and
	    // update the docs in GS collections node in the workspace tree
	    
	    if(operation == MOVE) {
		Utility.delete(sourceFolder);
                Gatherer.g_man.refreshWorkspaceTreeGreenstoneCollections();
		//WorkspaceTreeModel.refreshGreenstoneCollectionsNode();
	    }


	    progress_bar.setIndeterminate(false);
	    progress_bar.clear();
	    file_queue.getFileStatus().setText(Dictionary.get("FileActions.No_Activity"));
	    progress_bar.setString(Dictionary.get("FileActions.No_Activity"));
	}
    }

    private class FileTask
	extends Thread
    {
	private long id;
	private DragComponent source;
	private FileNode[] source_nodes;
	private DragComponent target;
	private FileNode target_node;
	private byte type;


	public FileTask(long id, DragComponent source, FileNode[] source_nodes, DragComponent target, FileNode target_node, byte type)
	{
	    this.id = id;
	    this.source = source;
	    this.source_nodes = source_nodes;
	    this.target = target;
	    this.target_node = target_node;
	    this.type = type;
	}


	public void run()
	{
	    // Reset the progress bar and set it to indeterminate while calculating its size
	    GProgressBar progress_bar = file_queue.getProgressBar();
	    progress_bar.reset();
	    progress_bar.setIndeterminate(true);

	    // Calculate the progress bar size
	    boolean cancelled = file_queue.calculateSize(source_nodes);
	    if (!cancelled) {
		file_queue.addJob(id, source, source_nodes, target, target_node, type);
		if (Gatherer.isGsdlRemote) {
		    String collection_name = CollectionManager.getLoadedCollectionName();

		    // Perform the appropriate action based on the job type (RemoteGreenstoneServer will queue)
		    if (type == FileJob.COPY) {
			// Copies: upload all the files at once in one zip file
			File[] source_files = new File[source_nodes.length];
			for (int i = 0; i < source_nodes.length; i++) {
			    source_files[i] = source_nodes[i].getFile();
			}
			Gatherer.remoteGreenstoneServer.uploadFilesIntoCollection(collection_name, source_files, target_node.getFile());
		    }
		    else if (type == FileJob.DELETE) {
			// Deletes: delete each top-level file/directory one at a time
			for (int i = 0; i < source_nodes.length; i++) {
			    Gatherer.remoteGreenstoneServer.deleteCollectionFile(collection_name, source_nodes[i].getFile());
			}
		    }
		    else if (type == FileJob.MOVE) {
			// Moves: move each top-level file/directory one at a time
			for (int i = 0; i < source_nodes.length; i++) {
			    Gatherer.remoteGreenstoneServer.moveCollectionFile(
                                     collection_name, source_nodes[i].getFile(), target_node.getFile());
			}
		    }
		}
	    }

	    progress_bar.setIndeterminate(false);
	    progress_bar.clear();
	}
    }

    public void explodeMetadataDatabase(File file)
    {
	// This must go in a separate thread because we need the progress bar to work (remote Greenstone server)
	Gatherer.invokeInEDT_replacesThreadStart("FileManager.explodeMetadataDatabase Task", Gatherer.ASYNC,
		new ExplodeMetadataDatabasePromptTask(file));
	//new ExplodeMetadataDatabasePromptTask(file).start();
    }

    // Works with replace_srcdoc_with_html.pl
    public void replaceSrcDocWithHtml(File[] files)
    {
	// This must go in a separate thread because we need the progress bar to work (remote Greenstone server)
	Gatherer.invokeInEDT_replacesThreadStart("FileManager.replaceSrcDocWithHTML task", Gatherer.ASYNC,
		new ReplaceSrcDocWithHtmlPromptTask(files));
	//new ReplaceSrcDocWithHtmlPromptTask(files).start();
    }

    private class ExplodeMetadataDatabasePromptTask
	extends Thread
    {
	private File metadata_database_file = null;

	public ExplodeMetadataDatabasePromptTask(File metadata_database_file)
	{
	    this.metadata_database_file = metadata_database_file;
	}

	public void run()
	{
	    ExplodeMetadataDatabasePrompt emp = new ExplodeMetadataDatabasePrompt(metadata_database_file);
	}
    }

    // Works with replace_srcdoc_with_html.pl
    private class ReplaceSrcDocWithHtmlPromptTask
	extends Thread
    {
	private File[] replace_these_srcdoc_files = null;

	public ReplaceSrcDocWithHtmlPromptTask(File[] replace_these_srcdoc_files)
	{
	    this.replace_these_srcdoc_files = replace_these_srcdoc_files;
	}

	public void run()
	{
	    ReplaceSrcDocWithHtmlPrompt prompt = new ReplaceSrcDocWithHtmlPrompt(replace_these_srcdoc_files);
	}
    }


    public void openFileInExternalApplication(File file)
    {
	// This must go in a separate thread because we need the progress bar to work (remote Greenstone server)
	new OpenFileInExternalApplicationTask(file).start();
    }


    private class OpenFileInExternalApplicationTask
	extends Thread
    {
	private File file = null;

	public OpenFileInExternalApplicationTask(File file)
	{
	    this.file = file;
	}

	public void run()
	{
	    // If we're using a remote Greenstone server, we need to download the file before viewing it...
	    if (Gatherer.isGsdlRemote) {
		// ... but only if it is inside the collection and we haven't already downloaded it
		if (file.getAbsolutePath().startsWith(Gatherer.getCollectDirectoryPath()) && file.length() == 0) {
		    if (Gatherer.remoteGreenstoneServer.downloadCollectionFile(
                                 CollectionManager.getLoadedCollectionName(), file).equals("")) {
			// Something has gone wrong downloading the file
			return;
		    }
		}
	    }
	    else if (Gatherer.isWebswing) {
		if (Gatherer.GS3) {
		    String full_filename = file.getAbsolutePath();
		    String collect_dirname = Gatherer.getCollectDirectoryPath();
		    
		    if (full_filename.startsWith(collect_dirname)) {
			// Strip off everything up to .../site/<site>/collect
			// => so local_filename start with the name of the actual collection
			
			String local_filename = full_filename.substring(collect_dirname.length());
			//String coll_name      = CollectionManager.getLoadedCollectionTailName();
			String opt_group_name = CollectionManager.getLoadedCollectionGroupName();
			
			//String this_collect_dirname = collect_dirname + coll_group_with_name;
			//if (full_filename.startsWith(this_collect_dirname)) {
			//String local_filename = full_filename.substring(this_collect_dirname.length());

			String library_url = Configuration.library_url.toString();
			String site = Configuration.site_name;
			
			
			//String library_collect_url = library_url + "/sites/" + site + "/collect/" + coll_group_with_name;
			String library_site_url = library_url + "/sites/" + site;
			
			String library_collect_url = library_site_url + "/collect"; 
			if (opt_group_name != null && !opt_group_name.equals("")) {
			    library_collect_url = library_collect_url + "/" + opt_group_name;
			}
			
			String library_collect_file_url = library_collect_url + "/" + local_filename;
			
			WebswingUtil.getWebswingApi().sendActionEvent("downloadURL", library_collect_file_url, null);
		    }
		    else {
			System.err.println("Warning - OpenFileInExternalApplicationTask.run() unable to construct a URL to access the following file" );
			System.err.println("    " + full_filename);
			System.err.println("Did not fall within " + collect_dirname);

			JOptionPane.showMessageDialog(Gatherer.g_man, Dictionary.get("FileActions.Webswing_Unable_To_Open", full_filename), Dictionary.get("General.Error"), JOptionPane.ERROR_MESSAGE);
		    }
		}
		else {
		    System.err.println("Error - Greenstone2 version of GLI using WebSwing does not currently support opening files externally" );
		    JOptionPane.showMessageDialog(Gatherer.g_man, Dictionary.get("FileActions.Webswing_Unable_To_Open_GS2"), Dictionary.get("General.Error"), JOptionPane.ERROR_MESSAGE);
		}
		    
		// finish here
		return;
	    }

	    // View the file in an external application
	    Gatherer.spawnApplication(file);
	}
    }


    public void newDummyDoc(DragTree tree, FileNode parent_node){
	newFolderOrDummyDoc(tree, parent_node, DUMMY_TYPE);
    }

    public void newEmptyFile(DragTree tree, FileNode parent_node){
	newFolderOrDummyDoc(tree, parent_node, FILE_TYPE);
    }


    public void newFolder(DragTree tree, FileNode parent_node) {
	newFolderOrDummyDoc(tree, parent_node, FOLDER_TYPE);
    }


    protected void newFolderOrDummyDoc(DragTree tree, FileNode parent_node, int type) {
	(new NewFolderOrDummyDocumentTask(tree, parent_node, type)).start();
    }


    private class NewFolderOrDummyDocumentTask
	extends Thread
    {
	private DragTree tree = null;
	private FileNode parent_node = null;
	private int type;

	// To run in EDT, need final or member vars. Reset this to null after use to free memory
	private NewFolderOrFilePrompt new_folder_prompt = null;
	
	public NewFolderOrDummyDocumentTask(DragTree tree, FileNode parent_node, int type)
	{
	    this.tree = tree;
	    this.parent_node = parent_node;
	    this.type = type;
	}

	public void run()
	{
	    // Ask the user for the folder/file name.
          String extension = ""; // for folders
          if (type == DUMMY_TYPE) extension =  ".nul";
          if (type == FILE_TYPE) extension = ".txt";

          // local variables referenced from an inner class must be final or effectively final
          // so we need to declare a new one and not change it
          String final_ext = extension;
	    Gatherer.invokeInEDT_replacesProceedInCurrThread(
	         "FileManager - NewFolderOrDummyDocumentTask.run()",
		 Gatherer.SYNC,
		 new Runnable() {
		     public void run() {
			 new_folder_prompt = new NewFolderOrFilePrompt(parent_node, type, final_ext);
			 //String name = new_folder_prompt.display();
			 new_folder_prompt.display();
			 new_folder_prompt.dispose();
		     }
		 });	
	    
	    String name = new_folder_prompt.getNameOfNewFolderOrFile();
	    new_folder_prompt = null; // important as it's now a member var, mustn't hog memory

	    // And if the name is non-null...
	    if (name != null) {
		FileSystemModel model = (FileSystemModel) tree.getModel();
		File folder_file = new File(parent_node.getFile(), name);

		//... check if it already exists.
		if (folder_file.exists()) { 
		    if (type == FILE_TYPE || type == DUMMY_TYPE) {
			JOptionPane.showMessageDialog(Gatherer.g_man, Dictionary.get("FileActions.File_Already_Exists_No_Create", name), Dictionary.get("General.Error"), JOptionPane.ERROR_MESSAGE);
		    }
		    else {
			JOptionPane.showMessageDialog(Gatherer.g_man, Dictionary.get("FileActions.Folder_Already_Exists", name), Dictionary.get("General.Error"), JOptionPane.ERROR_MESSAGE);
		    }
		}
		// Otherwise create it.
		else {
		    try {
			if (type == FILE_TYPE || type == DUMMY_TYPE) {
			    folder_file.createNewFile();
			    if (Gatherer.isGsdlRemote) {
				Gatherer.remoteGreenstoneServer.uploadCollectionFile(
                                         CollectionManager.getLoadedCollectionName(), folder_file);
			    }
			}
			else {
			    folder_file.mkdirs();
			    if (Gatherer.isGsdlRemote) {
				Gatherer.remoteGreenstoneServer.newCollectionDirectory(
                                         CollectionManager.getLoadedCollectionName(), folder_file);
			    }
			}

			// Update the parent node to show the new folder
			parent_node.refresh();

			// Refresh workspace tree (collection tree is done automatically)
			Gatherer.g_man.refreshWorkspaceTree(DragTree.COLLECTION_CONTENTS_CHANGED);
		    }
		    catch (Exception exception) {
			if (type == FILE_TYPE || type == DUMMY_TYPE) {
			    JOptionPane.showMessageDialog(Gatherer.g_man, Dictionary.get("FileActions.File_Create_Error", name), Dictionary.get("General.Error"), JOptionPane.ERROR_MESSAGE);
			}
			else {
			    JOptionPane.showMessageDialog(Gatherer.g_man, Dictionary.get("FileActions.Folder_Create_Error", name), Dictionary.get("General.Error"), JOptionPane.ERROR_MESSAGE);
			}
		    }
		}
		
		folder_file = null;
		model = null;
	    }
	    name = null;
	}
    }


    public void renameCollectionFile(DragTree collection_tree, FileNode collection_tree_node)
    {
	// This must go in a separate thread because we need the progress bar to work (remote Greenstone server)
	new RenameTask(collection_tree, collection_tree_node).start();
    }


    private class RenameTask
	extends Thread
    {
	private DragTree collection_tree = null;
	private FileNode collection_tree_node = null;

	// To run in EDT, need final or member variables. Reset this variable to null after
	// use, to free memory (signals the garbage collector)
	RenamePrompt rename_prompt = null;
	
	public RenameTask(DragTree collection_tree, FileNode collection_tree_node)
	{
	    this.collection_tree = collection_tree;
	    this.collection_tree_node = collection_tree_node;
	}

	public void run()
	{
	    Gatherer.invokeInEDT_replacesProceedInCurrThread(
	         "FileManager - RenameTask.run()",
		 Gatherer.SYNC,
		 new Runnable() {
		     public void run() {
			 rename_prompt = new RenamePrompt(collection_tree_node);
			 rename_prompt.display();
		     }
		 });
	    
	    String new_collection_file_name = rename_prompt.getNewName();
	    rename_prompt.dispose();
	    rename_prompt = null; // important as it's now a member var to not hog memory

	    if (new_collection_file_name != null) {
		File collection_file = collection_tree_node.getFile();
		File new_collection_file = new File(collection_file.getParentFile(), new_collection_file_name);
                FileNode new_collection_tree_node;
                
                if (collection_tree instanceof CollectionTree) {
                  new_collection_tree_node = new CollectionTreeNode(new_collection_file);
                } else {
                  new_collection_tree_node = new FullCollectionTreeNode(new_collection_file);
                }
		file_queue.addJob(System.currentTimeMillis(), collection_tree, new FileNode[] { collection_tree_node }, collection_tree, new_collection_tree_node, FileJob.RENAME);
		if (Gatherer.isGsdlRemote) {
		    Gatherer.remoteGreenstoneServer.moveCollectionFile(
                             CollectionManager.getLoadedCollectionName(), collection_file, new_collection_file);
		}
	    }
	}
    }

    public void newCollectionFile(DragTree collection_tree, FileNode parent_node)
    {
	new ReplaceOrNewTask(collection_tree, parent_node, false ).start();
    }

    public void replaceCollectionFile(DragTree collection_tree, FileNode collection_tree_node)
    {
	// This must go in a separate thread because we need the progress bar to work (remote Greenstone server)
	new ReplaceOrNewTask(collection_tree, collection_tree_node, true).start();
    }


    private class ReplaceOrNewTask
	extends Thread implements FileCopiedSuccessListener
    {
	private DragTree collection_tree = null;
	private FileNode collection_tree_node = null;
	private boolean replacing = false;

	public ReplaceOrNewTask(DragTree collection_tree, FileNode collection_tree_node, boolean replacing)
	{
	    this.collection_tree = collection_tree;
	    this.collection_tree_node = collection_tree_node; // either the file to replace, or the folder to go into
	    this.replacing = replacing;
	}

	public void run()
	{
	    JFileChooser file_chooser = new JFileChooser(startup_directory);
	    if (replacing) {
		file_chooser.setDialogTitle(Dictionary.get("ReplacePrompt.Title"));
	    } else {
		file_chooser.setDialogTitle(Dictionary.get("NewPrompt.Title"));
	    }
	    File new_file = null;
	    int return_val = file_chooser.showOpenDialog(null);
	    if(return_val == JFileChooser.APPROVE_OPTION) {
		new_file = file_chooser.getSelectedFile();
	    }
	    
	    if (new_file == null) {
		return;
	    }
	    
	    // save the search path for next time
	    startup_directory = new_file.getParentFile();
	    // make up a node for the file to bring in
	    WorkspaceTreeNode source_node = new WorkspaceTreeNode(new_file);

	    //DebugStream.setDebugging(true, "FileManager.ReplaceTask");

	    // Some different handling if the old and new tail file names are the same and target file goes into the same
	    // location in collection tree as source.
	    // This avoids past errors upon replacing with same filename (diff file contents) where the attached meta gets lost,
	    // or remote file gets updated but file gone missing in client-GLI view until collection reopened.
	    boolean isSameLeafName = false;
	    if(replacing && collection_tree_node.getFile().getName().equals(new_file.getName())) {
		DebugStream.println(" @@@ File Replace: New file has the same name as existing.");
		isSameLeafName = true;
	    }
	    
	    File target_directory;
	    FileNode parent; // store the original source's parent, need it several times after changing source
	    if (replacing) {
		target_directory = collection_tree_node.getFile().getParentFile();
		parent = (FileNode)collection_tree_node.getParent();
	    } else {
		target_directory = collection_tree_node.getFile();
		parent = collection_tree_node;
	    }
            
            FileNode new_collection_tree_node;
            if (collection_tree instanceof CollectionTree) {
              new_collection_tree_node = new CollectionTreeNode(new File(target_directory, new_file.getName()));
            } else {
             new_collection_tree_node = new FullCollectionTreeNode(new File(target_directory, new_file.getName()));
            } 
	    
	    //FileNode parent = (FileNode)collection_tree_node.getParent(); // store the original source's parent, need it several times after changing source

	    if(isSameLeafName) {
		// If the file name of the replacing file IS the same as the one being replaced
		// perform a COPY operation, which will copy across metadata too, after confirming whether the user really wants to replace the source with identically named target

		// (a) First, this instance of ReplaceTask and no other starts listening to whether the user
		// DIDN'T CANCEL out of an identical filename copy operation and if this local file copy
		// was a success. If so, on successful file copy event fired (only then), the source file
		// from the workspace tree will also be uploaded to the remote GS3 
		file_queue.addFileCopiedSuccessListener(this); 

		// (b) Now can finally add the COPY job to the queue
		file_queue.addJob(System.currentTimeMillis(), Gatherer.g_man.gather_pane.workspace_tree, new FileNode[] { source_node }, collection_tree, parent, FileJob.COPY);
		
	    } else {		
		// If the file name of the replacing file is NOT the same as the one being replaced:
		// (a) Again, this ReplaceTask instance needs to listen for the file copy event fired,
		// so that the source file will also get uploaded to the remote GS3 on FileCopiedSuccess
		file_queue.addFileCopiedSuccessListener(this);

		// (b) copy the new file in - but don't bring metadata
		file_queue.addJob(System.currentTimeMillis(), Gatherer.g_man.gather_pane.workspace_tree, new FileNode[] { source_node }, collection_tree, parent, FileJob.COPY_FILE_ONLY);
	
		if (replacing) {
		    // (c) final step to finish off: do a replace of old file with new file
		    file_queue.addJob(System.currentTimeMillis(), collection_tree, new FileNode[] { collection_tree_node }, collection_tree, new_collection_tree_node, FileJob.REPLACE);
		}
	    }



	    //DebugStream.setDebugging(false, "FileManager.ReplaceTask");
	}
    
	
	/** In order to detect that the user cancelled out of replacing an identically named target file, 
	 * we now listen to events fired that the file was successfully copied across. Only then do we
	 * bother transferring the source file (from the workspace) into the target location in the
	 * collection on the remote file system. We don't do this if the user cancelled.
	 */
	public void fileCopiedSuccessfully(File new_file) {	    
	    
	    //DebugStream.setDebugging(true, "FileManager.ReplaceTask.fileCopiedSuccessfully");
	    
	    if (Gatherer.isGsdlRemote) {
		File target_directory;
		if (this.replacing) {
		    target_directory = this.collection_tree_node.getFile().getParentFile();
		} else {
		    target_directory = this.collection_tree_node.getFile();
		}
		File collection_tree_node_file = this.collection_tree_node.getFile();
		
		String collection_name = CollectionManager.getLoadedCollectionName();
		if (this.replacing) {
		Gatherer.remoteGreenstoneServer.deleteCollectionFile(collection_name, collection_tree_node_file);
		}
		Gatherer.remoteGreenstoneServer.uploadFilesIntoCollection(collection_name, new File[] { new_file }, target_directory);
	    }

	    // stop listening to further events fired now that we've handled this event successfully
	    file_queue.removeFileCopiedSuccessListener(this);
	    //DebugStream.setDebugging(false, "FileManager.ReplaceTask.fileCopiedSuccessfully");
	}
    }
}
