/**
 *#########################################################################
 *
 * A component of the Gatherer application, part of the Greenstone digital
 * library suite from the New Zealand Digital Library Project at the
 * University of Waikato, New Zealand.
 *
 * Author: John Thompson, Greenstone Digital Library, University of Waikato
 *
 * Copyright (C) 1999 New Zealand Digital Library Project
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *########################################################################
 */
package org.greenstone.gatherer.cdm;

import java.awt.*;
import java.awt.event.*;
import java.util.*;
import javax.swing.*;
import javax.swing.event.*;
import org.greenstone.gatherer.Configuration;
import org.greenstone.gatherer.DebugStream;
import org.greenstone.gatherer.Dictionary;
import org.greenstone.gatherer.Gatherer;
import org.greenstone.gatherer.gui.DesignPaneHeader;
import org.greenstone.gatherer.gui.GComboBox;
import org.greenstone.gatherer.gui.GLIButton;
import org.greenstone.gatherer.metadata.MetadataElement;
import org.greenstone.gatherer.metadata.MetadataSetManager;
import org.greenstone.gatherer.util.CheckList;
import org.greenstone.gatherer.util.JarTools;
import org.greenstone.gatherer.util.StaticStrings;


/** This class is responsible for coordinating the IndexManager and IndexOptionManager (controls levels, stem options etc) and BuildTypeManager into one panel  
 * @author John Thompson, Greenstone Digital Library, University of Waikato
 * @version 2.3
 */
public class IndexingManager {

    private IndexOptionManager option_manager = null;
    private SearchIndexManager index_manager = null;
    private BuildTypeManager build_type_manager = null;
    private SortFieldManager sortfield_manager = null;
    private FacetManager facet_manager = null;

    private String build_type = null;

    private Control controls = null;

    public IndexingManager()
    {
	build_type_manager = new BuildTypeManager();
	build_type = build_type_manager.getBuildType();
	option_manager = new IndexOptionManager(build_type);
	if (isMG()) {
	    index_manager = new SearchIndexManager(CollectionDesignManager.collect_config.getMGIndexes(), build_type);
	} else {
	    index_manager = new SearchIndexManager(CollectionDesignManager.collect_config.getMGPPIndexes(), build_type);
	}
	// we always create the managers, but only display the controls if we
	// are Lucene/SOLR
	sortfield_manager = new SortFieldManager(CollectionDesignManager.collect_config.getSorts(), build_type);
	facet_manager = new FacetManager(CollectionDesignManager.collect_config.getFacets(), build_type);

    }
    public boolean isMGPP() {
	return build_type_manager.isMGPP();
    }

    public boolean isLucene() {
	return build_type_manager.isLucene();
    }

    public boolean isMG() {
	return build_type_manager.isMG();
    }

    public boolean isSOLR() {
	return build_type_manager.isSOLR();
    }

    public int getNumLevels() {
	return option_manager.getNumLevels();
    }

    public int getNumIndexes() {
	return index_manager.getSize();
    }

    public ArrayList getIndexes() {
	return index_manager.getIndexes();
    }

    public ArrayList getLevels() {
      if (!isMG()) {
	    return option_manager.getLevels();
	}
	return null;
    }

  public ArrayList getSortFields() {
    if (isLucene() || isSOLR()) {
      return sortfield_manager.getIndexes();
    }
    return null;
  }
  public ArrayList getFacets() {
    if (isSOLR()) {
      return facet_manager.getIndexes();
    }
    return null;
  }

  
    public Control getControls() {
	if (controls == null) {
	    controls = new IndexingControl();
	}
	return controls;
    }

    public void destroy() {

    }

    /** Called when the detail mode has changed which in turn may cause several design elements to be available/hidden
     * @param mode the new mode as an int
     */
    public void modeChanged(int mode) {

    }


    private class IndexingControl
	extends JPanel
	implements Control, BuildTypeManager.BuildTypeListener
    {
	JPanel main_index_pane = null;
	JPanel index_options_panel = null;

	JPanel index_sort_facet_panel = null;
    // these get added/removed depending on indexer in use
    private JPanel sortfield_panel = null;
    private JPanel facet_panel = null;

	public IndexingControl() {
	    super();
            this.setComponentOrientation(Dictionary.getOrientation());
	    // Creation
	    JPanel header_pane = new DesignPaneHeader("CDM.GUI.Indexes", "searchindexes");	   
	    
	    JPanel build_type_panel = (JPanel)build_type_manager.getControls();
	    index_options_panel = (JPanel)option_manager.getControls();

	    index_sort_facet_panel = new JPanel();
	    
	    index_sort_facet_panel.setLayout(new GridLayout(0,1,0,5));
	    JPanel index_panel = (JPanel)index_manager.getControls(); 
	    index_sort_facet_panel.add(index_panel);
	    if (isLucene() || isSOLR() ) {
		sortfield_panel = (JPanel)sortfield_manager.getControls();
		index_sort_facet_panel.add(sortfield_panel);
	    }
	    if (isSOLR()) {
		facet_panel = (JPanel)facet_manager.getControls();
		index_sort_facet_panel.add(facet_panel);
            }
	    main_index_pane = new JPanel();
	    main_index_pane.setLayout(new BorderLayout());
	    main_index_pane.add(build_type_panel, BorderLayout.NORTH);
	    main_index_pane.add(index_sort_facet_panel, BorderLayout.CENTER);
	    main_index_pane.add(index_options_panel, BorderLayout.SOUTH);
	    main_index_pane.setComponentOrientation(Dictionary.getOrientation());
            
	    setBorder(BorderFactory.createEmptyBorder(0,5,0,0));
	    setLayout(new BorderLayout());
	    add(header_pane, BorderLayout.NORTH);
	    add(main_index_pane, BorderLayout.CENTER);

	    build_type_manager.addBuildTypeListener(this);
	    build_type_manager.addBuildTypeListener(index_manager);
	    build_type_manager.addBuildTypeListener(sortfield_manager);
	    build_type_manager.addBuildTypeListener(facet_manager);
	    build_type_manager.addBuildTypeListener(option_manager);
	}

	public void loseFocus() {}
	public void gainFocus() {}
	public void destroy() {}

	public void buildTypeChanged(String new_build_type)
	{
	    if (build_type.equals(new_build_type)) {
		// shouldn't happen
		return;
	    }
	    if (hasSorts(new_build_type) && !hasSorts(build_type)) {
		sortfield_panel = (JPanel)sortfield_manager.getControls();
		index_sort_facet_panel.add(sortfield_panel);
		// add sort panel
	    }
	    if (hasSorts(build_type) && ! hasSorts(new_build_type)) {
		// remove sort panel
		index_sort_facet_panel.remove(sortfield_panel);
	    } 
	    if (hasFacets(new_build_type) && ! hasFacets(build_type)) {
		// add facet
		facet_panel = (JPanel)facet_manager.getControls();
		index_sort_facet_panel.add(facet_panel);
	    } 
	    if (hasFacets(build_type) && ! hasFacets(new_build_type)) {
		// remove facet pane
		index_sort_facet_panel.remove(facet_panel);
	    }
 	    build_type = new_build_type;
	}

	private boolean hasSorts(String build_type) {
	    return build_type.equals(BuildTypeManager.BUILD_TYPE_LUCENE) || build_type.equals(BuildTypeManager.BUILD_TYPE_SOLR);
	}
	private boolean hasFacets(String build_type) {
	    return build_type.equals(BuildTypeManager.BUILD_TYPE_SOLR);

	}

    }
}
