/**
 *#########################################################################
 *
 * A component of the Gatherer application, part of the Greenstone digital
 * library suite from the New Zealand Digital Library Project at the
 * University of Waikato, New Zealand.
 *
 * <BR><BR>
 *
 * Author: John Thompson, Greenstone Digital Library, University of Waikato
 *
 * <BR><BR>
 *
 * Copyright (C) 1999 New Zealand Digital Library Project
 *
 * <BR><BR>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * <BR><BR>
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * <BR><BR>
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *########################################################################
 */
package org.greenstone.gatherer.gui.metaaudit;

import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.event.*;
import javax.swing.tree.*;
import org.greenstone.gatherer.Configuration;
import org.greenstone.gatherer.Dictionary;
import org.greenstone.gatherer.Gatherer;
import org.greenstone.gatherer.collection.CollectionTreeNode;
import org.greenstone.gatherer.gui.GLIButton;
import org.greenstone.gatherer.gui.SimpleMenuBar;
import org.greenstone.gatherer.gui.ModalDialog;


/** The MetaAuditFrame provides a table view of all of the metadata assigned to a selection of CollectionTreeNodes. All values for a certain file and a certain metadata element appear in the same cell. This table can be sorted by any column, and also has a MS Excel-like AutoFilter allowing you to restrict the rows visible to only those that match a certain set of criteria (applied to each column, and then 'ANDED', or cojoined, to determine the filter). Finally this dialog does not block the Gatherer tool, so the file selection can be changed and the dialog will just generate a new table dynamically.<BR>
 * Much effort has gone into optimizing this table, as it quickly becomes slow and unresponsive to build/filter/sort when the number of records selected is high. However its performance is still nowhere as good as the Excel spreadsheet, and selections of 1000+ records can cause some serious waiting problems. Performance progression, shown in terms of time taken to perform action, are shown below. Note that building includes both the creation of the data model, and/or the time taken to lay out row and column sizes (due to this using multiple entry cells). Also all tables, by default, must be sorted by one column, which is initially the first column in ascending order:<BR><BR>
 * <TABLE border=1 cellspacing=0 cellpadding=5>
 * <TR><TD align=center colspan=5>Time Taken with 1000 records (ms)</TD></TR>
 * <TR><TD>Action</TD><TD>Original</TD><TD>Revision 1</TD><TD>Revision 2</TD><TD>Revision 3</TD></TR>
 * <TR><TD>Build</TD><TD align=right>113013</TD><TD align=right>1500</TD><TD align=right>1280</TD><TD align=right>665</TD></TR>
 * <TR><TD>Sort</TD><TD align=right>34228</TD><TD align=right>353</TD><TD align=right>341</TD><TD align=right>338</TD></TR>
 * <TR><TD>Filter</TD><TD align=right>57110</TD><TD align=right>485</TD><TD align=right>485</TD><TD align=right>485</TD></TR>
 * </TABLE>
 * <BR><BR>
 * <TABLE border=1 cellspacing=0 cellpadding=5>
 * <TR><TD align=center colspan=5>Time Taken with 10,000 records (ms) [hdlsub]</TD></TR>
 * <TR><TD>Action</TD><TD>Original</TD><TD>Revision 1</TD><TD>Revision 2</TD><TD>Revision 3</TD></TR>
 * <TR><TD>Build</TD><TD align=right>DNF</TD><TD align=right>8667</TD><TD align=right>6784</TD><TD align=right>3081</TD></TR>
 * <TR><TD>Sort</TD><TD align=right>DNF</TD><TD align=right>236</TD><TD align=right>241</TD><TD align=right>228</TD></TR>
 * <TR><TD>Filter</TD><TD align=right>DNF</TD><TD align=right>59272</TD><TD align=right>59767</TD><TD align=right>59614</TD></TR>
 * </TABLE>
 * <BR><BR>
 * <TABLE border=1 cellspacing=0 cellpadding=5>
 * <TR><TD align=center colspan=5>Time Taken with 30,000 records (ms) [hdl]</TD></TR>
 * <TR><TD>Action</TD><TD>Original</TD><TD>Revision 1</TD><TD>Revision 2</TD><TD>Revision 3</TD></TR>
 * <TR><TD>Build</TD><TD align=right>DNF</TD><TD align=right>19037</TD><TD align=right>16478</TD><TD align=right>8111</TD></TR>
 * <TR><TD>Sort</TD><TD align=right>DNF</TD><TD align=right>314</TD><TD align=right>301</TD><TD align=right>385</TD></TR>
 * <TR><TD>Filter</TD><TD align=right>DNF</TD><TD align=right>[300000]</TD><TD align=right>[300000]</TD><TD align=right>[300000]</TD></TR>
 * </TABLE><BR><BR>
 * It is easy to see that while building and sorting have become quite fast and reasonable, sorting is still unacceptable in terms of performance. The difficulty is this, that while sorting is easy (and fast) to implement via the Decorator pattern (making the overall table model a MDVC one) it requires the entire contents of a column to be available. Thus no matter the speed of your sort algorithm you will end up making N calls to getValueAt() in the table data model, for N records. This is not a problem for the initial model, hence the small sort times. However when you attempt to place another deocrator model (or modify the existing one) in order to filter only selected rows, even if the filtering itself takes little time, this little time quickly grows to unacceptable levels. In fact testing has shown it to grow much faster than linear time (though I'm not sure where). Doubling the number of records to filter increases the processing time by 5-7 times.<BR>
 * In attempting to solve this problem, I have found many resources that mention a similar problem when trying to build visual representations of database tables, especially over networks or other temporally-non-deterministic connections. The common solution is to 'page' only those records necessary into memory on demand, thus making initial response significantly faster, while paying a small price for subsequent page seeks. There are even studies and algorithms for determining pages loads etc, I would imagine garnered from the memory management community. However none of these really help me usless I can get around the problem of having to have the entire columns population immeditaly accessable for sorting purposes.<BR>
 * I believe with more time is would be possible to make a hybrid table model with properties such as you would require for the aforemenetioned database application, but which also provides fast methods for determining the highest and lowest records perhaps by building traversable heaps in the original model. Such a solution would incur a build penalty, and could consume significant memory, but would provide ordered and sequential access to the data within.<BR>
 * The discussion aside, I have run out of time to do anything further on this, so will include as is, and provide a warning dialog whenever the number of records might suffer slow processing (1000+ records).
 * @author John Thompson, Greenstone Digital Library, University of Waikato
 * @version 2.3
 */
public class MetaAuditFrame
    extends ModalDialog 
    implements TreeSelectionListener {
    public AutofilterDialog autofilter_dialog;
    /** Whether the selection has changed since the last time we built the model (because it has been hidden and theres no point refreshing a model you can't see!). */
    private boolean invalid = true;
    /** An array holding the most recent list of paths selected (plus some nulls for those paths removed). */
    private CollectionTreeNode[] records = null;
    /** A reference to ourselves so that inner classes can interact with us. */
    private MetaAuditFrame self;
    /** The table in which the metadata is shown. */
    private MetaAuditTable table;
    /** The default size for the metaaudit dialog. */
    static final private Dimension SIZE = new Dimension(640,505);


    public MetaAuditFrame()
    {
	super(Gatherer.g_man);
        
	// Arguments
	this.autofilter_dialog = new AutofilterDialog(this);
	this.self = this;
	this.table = new MetaAuditTable(this);
        this.setComponentOrientation(Dictionary.getOrientation());
	// Creation
	setDefaultCloseOperation(JDialog.HIDE_ON_CLOSE);
	setSize(SIZE);
	setTitle(Dictionary.get("MetaAudit.Title"));
	setJMenuBar(new SimpleMenuBar("reviewingmetadata")); 
	
	JPanel content_pane = (JPanel) getContentPane();
        content_pane.setComponentOrientation(Dictionary.getOrientation());
	JPanel button_pane = new JPanel();
        button_pane.setComponentOrientation(Dictionary.getOrientation());
        
	JButton close_button = new GLIButton(Dictionary.get("MetaAudit.Close"), Dictionary.get("MetaAudit.Close_Tooltip"));
	
	// Connection
	close_button.addActionListener(new CloseListener());
	Gatherer.c_man.getCollectionTree().addTreeSelectionListener(this);

	// Layout
	button_pane.setBorder(BorderFactory.createEmptyBorder(5,5,5,5));
	button_pane.setLayout(new BorderLayout());
	button_pane.add(close_button, BorderLayout.CENTER);
		 
	content_pane.setBorder(BorderFactory.createEmptyBorder(5,5,5,5));
	content_pane.setLayout(new BorderLayout());
	content_pane.add(new JScrollPane(table), BorderLayout.CENTER);
	content_pane.add(button_pane, BorderLayout.SOUTH);
		  
	Dimension screen_size = Configuration.screen_size;
	setLocation((screen_size.width - SIZE.width) / 2, (screen_size.height - SIZE.height) / 2);
	screen_size = null;
	close_button = null;
	button_pane = null;
	content_pane = null;
    }

    /** Destructor. */
    public void destroy() {
	records = null;
	self = null;
	table = null;
    }

    /** Display the dialog on screen. */
    public void display() {
	if(invalid) {
	    rebuildModel();
	}
	setVisible(true);
    }

    /** This method is called whenever the selection within the collection tree changes. This causes the table to be rebuilt with a new model.
     * @param event A <strong>TreeSelectionEvent</strong> containing information about the event.
     */
    public void valueChanged(TreeSelectionEvent event) {
	Object source = event.getSource();
	if(source instanceof JTree) {
	    TreePath paths[] = ((JTree)source).getSelectionPaths();
	    if(paths != null) {
		records = new CollectionTreeNode[paths.length];
		for(int i = 0; i < paths.length; i++) {
		    records[i] = (CollectionTreeNode) paths[i].getLastPathComponent();
		}
		if(isVisible()) {
		    rebuildModel();
		}
		else {
		    invalid = true;
		}
	    }
	}
    }

    public void wait(boolean waiting) {
	Component glass_pane = getGlassPane();
	if(waiting) {
	    // Show wait cursor.
	    glass_pane.setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
	    glass_pane.setVisible(true);
	}
	else {
	    // Hide wait cursor.
	    glass_pane.setVisible(false);
	    glass_pane.setCursor(Cursor.getPredefinedCursor(Cursor.DEFAULT_CURSOR));
	}
	glass_pane = null;
    }

    /** Rebuild the metaaudit table model using the current collection tree selection. 
     */
    private void rebuildModel() {
	// Build and set model
	table.newModel(records);
	// Done.
	invalid = false;
    }

    /** Listens for actions upon the close button, and if detected closes the dialog */
    private class CloseListener
	implements ActionListener {
	/** Any implementation of ActionListener must include this method so that we can be informed when an action has been performed on our target control(s).
	 * @param event An <strong>ActionEvent</strong> containing information about the event.
	 */
	public void actionPerformed(ActionEvent event) {
	    self.setVisible(false);
	}
    }
}
