/**
 *#########################################################################
 *
 * A component of the Gatherer application, part of the Greenstone digital
 * library suite from the New Zealand Digital Library Project at the
 * University of Waikato, New Zealand.
 *
 * <BR><BR>
 *
 * Author: Shaoqun Wu, Greenstone Digital Library, University of Waikato
 *
 * <BR><BR>
 *
 * Copyright (C) 2006 New Zealand Digital Library Project
 *
 * <BR><BR>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * <BR><BR>
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * <BR><BR>
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *########################################################################
 */
package org.greenstone.gatherer.gems;

import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.event.*;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Collection;
import java.util.Vector;
import java.io.File;


import org.greenstone.gatherer.Configuration;
import org.greenstone.gatherer.Dictionary;
import org.greenstone.gatherer.gui.ModalDialog;
import org.greenstone.gatherer.gui.GLIButton;
import org.greenstone.gatherer.gui.TestingPreparation;

public class OpenMetadataSetPrompt 
    extends ModalDialog {

    static private Dimension SIZE = new Dimension(500, 500);
        
    private ArrayList available_metadata_sets;
    private ArrayList listeners; 

    private JButton open_button = null;
    private JButton cancel_button = null; 
    private JButton browse_button = null; 
    private JList  available_set_list = null;
    private OpenMetadataSetPrompt self;
    private MetadataSetManager meta_manager;
    private JTextArea description_textarea = null;

    private boolean cancelled = false;
    
    public OpenMetadataSetPrompt(Frame parent,MetadataSetManager msm) {
	super(parent, true);
        
        self = this;
	meta_manager = msm;
	setSize(SIZE);
	setTitle(Dictionary.get("GEMS.OpenMetadataSetPrompt.Title"));
	listeners = new ArrayList();

	JPanel content_pane = (JPanel) getContentPane();
        content_pane.setOpaque(true);
	content_pane.setComponentOrientation(Dictionary.getOrientation());
        
	JLabel  available_metadata_sets_label = new JLabel(Dictionary.get("GEMS.OpenMetadataSetPrompt.Available_Sets"));
	available_metadata_sets_label.setOpaque(true);
        available_metadata_sets_label.setComponentOrientation(Dictionary.getOrientation());
        
        available_set_list = new JList();
	available_set_list.setCellRenderer(new MetadatSetListCellRenderer());
	available_set_list.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
        available_set_list.setFixedCellHeight(20);
	available_set_list.addListSelectionListener(new MetadataSetListSelectionListener());
        available_set_list.setComponentOrientation(Dictionary.getOrientation());
        
        JPanel set_pane = new JPanel();
        set_pane.setComponentOrientation(Dictionary.getOrientation());
        set_pane.setLayout(new BorderLayout());
        set_pane.add(available_metadata_sets_label,BorderLayout.NORTH);
	set_pane.add(new JScrollPane(available_set_list),BorderLayout.CENTER);
        
	JLabel  metadata_set_des_label = new JLabel(Dictionary.get("GEMS.Set_Description"));
	metadata_set_des_label.setOpaque(true);
        metadata_set_des_label.setComponentOrientation(Dictionary.getOrientation());
        
	description_textarea = new  JTextArea();
        description_textarea.setOpaque(true);
        description_textarea.setEditable(false);
        description_textarea.setLineWrap(true); 
        description_textarea.setWrapStyleWord(true);
        description_textarea.setComponentOrientation(Dictionary.getOrientation());
        
	JPanel des_pane = new JPanel();
        des_pane.setLayout(new BorderLayout());
        des_pane.add(metadata_set_des_label,BorderLayout.NORTH);
	des_pane.add(new JScrollPane(description_textarea),BorderLayout.CENTER);
        des_pane.setComponentOrientation(Dictionary.getOrientation());
        
	JPanel button_pane = new JPanel();
        button_pane.setComponentOrientation(Dictionary.getOrientation());
        
	open_button = new GLIButton(Dictionary.get("GEMS.OpenMetadataSetPrompt.Open"), Dictionary.get("GEMS.OpenMetadataSetPrompt.Open_Tooltip"));
	open_button.setEnabled(true);	
        
	browse_button = new GLIButton(Dictionary.get("General.Browse"));
        browse_button.setEnabled(true);

	cancel_button = new GLIButton(Dictionary.get("General.Cancel"), Dictionary.get("General.Cancel_Tooltip"));
	cancel_button.setEnabled(true);


	// Add listeners
	open_button.addActionListener(new ActionListener() {
		public void actionPerformed(ActionEvent event) {
		    notifyListeners();
		    self.dispose();
		}
	    });

	browse_button.addActionListener(new ActionListener() {
		public void actionPerformed(ActionEvent event) {
		    String collect_dir = meta_manager.getCollectionPath();
		    JFileChooser file_chooser =  new JFileChooser(new File(collect_dir));
		    file_chooser.setMultiSelectionEnabled(false);
		    int result = file_chooser.showOpenDialog(null);
		    if (result == JFileChooser.APPROVE_OPTION){
			String file_path = file_chooser.getSelectedFile().toString();
			MetadataSetInfo meta_info = meta_manager.getMetadataSet(file_path);
			meta_info.setNew(false);
			MetadataSetEvent mse = new MetadataSetEvent(meta_info);
			for(int i=0;i<listeners.size();i++){
			    MetadataSetListener msl = (MetadataSetListener)listeners.get(i);
			    msl.metadataSetChanged(mse);          
			}
			
		    }     
		    notifyListeners();
		    self.dispose();
		}
	    });



	cancel_button.addActionListener(new ActionListener() {
		public void actionPerformed(ActionEvent event) {
		    cancelled = true;
		    self.dispose();
		}
	    });


	button_pane.setLayout(new GridLayout(1,3));
	button_pane.add(open_button);
	button_pane.add(browse_button);
	button_pane.add(cancel_button);
	

	content_pane.setLayout(new BorderLayout());
	content_pane.add(set_pane, BorderLayout.NORTH);
	content_pane.add(des_pane, BorderLayout.CENTER);
	content_pane.add(button_pane, BorderLayout.SOUTH);
	
	// Show
	Dimension screen_size = Configuration.screen_size;
	setLocation((screen_size.width - SIZE.width) / 2, (screen_size.height - SIZE.height) / 2);
	setVisible(false);

    }


    public void display(){
	cancelled = false;
        available_metadata_sets = meta_manager.getAvailableMetadataSets();
	available_set_list.setListData(new Vector((Collection)available_metadata_sets));

	TestingPreparation.setNamesRecursively(this);
        setVisible(true);          
    }

    public void addMetadataSetListener(MetadataSetListener msl){
        listeners.add(msl);
   
    }

    public boolean isCancelled() {
	return cancelled;
    }

    private void notifyListeners(){
	Object selectedValue = available_set_list.getSelectedValue();
	
	if (selectedValue !=null && (selectedValue instanceof MetadataSetInfo)){
	    MetadataSetEvent mse = new MetadataSetEvent((MetadataSetInfo)selectedValue);
	    for(int i=0;i<listeners.size();i++){
		MetadataSetListener msl = (MetadataSetListener)listeners.get(i);
		msl.metadataSetChanged(mse);          
	    }
	}
    }

    public void openMetadataSet(String metadata_path){
       MetadataSetInfo set_info = meta_manager.getMetadataSet(metadata_path);                       
	notifyListeners(set_info);   
    }

    private void notifyListeners(MetadataSetInfo set_info){
	MetadataSetEvent mse = new MetadataSetEvent(set_info);
	for(int i=0;i<listeners.size();i++){
	    MetadataSetListener msl = (MetadataSetListener)listeners.get(i);
	    msl.metadataSetChanged(mse);          
	}	
    }

    private class MetadatSetListCellRenderer extends JLabel implements ListCellRenderer {
	public  MetadatSetListCellRenderer() {
	    setOpaque(true);
	}
	
     public Component getListCellRendererComponent(
						   JList list,
						   Object value,
						   int index,
						   boolean isSelected,
         boolean cellHasFocus)
	{
	    String name= "unknown";

	    if (value instanceof MetadataSetInfo){
		MetadataSetInfo meta_info = (MetadataSetInfo) value;
		name = meta_info.getMetadataSetName(); //get the name of metadata set
	    }
  
            setText(name);
            if (isSelected) {
		setBackground(list.getSelectionBackground());
		setForeground(list.getSelectionForeground());
	    }
	    else {
		setBackground(list.getBackground());
		setForeground(list.getForeground());
	    }
	              
	    return this;
	}
    }
   
    
    private class MetadataSetListSelectionListener implements ListSelectionListener  {
        public void valueChanged(ListSelectionEvent lse){
	    if (lse.getValueIsAdjusting()) return;

	    Object selectedValue = available_set_list.getSelectedValue();

            if (selectedValue !=null && (selectedValue instanceof MetadataSetInfo)){
                MetadataSetInfo meta_info = (MetadataSetInfo)selectedValue; 
		String des = meta_info.getMetadataSetDescription(); //get the description of the metadata set
                description_textarea.setText(des);
	    }      

	}
    }
   
 
}
