/**
 *#########################################################################
 *
 * A component of the Gatherer application, part of the Greenstone digital
 * library suite from the New Zealand Digital Library Project at the
 * University of Waikato, New Zealand.
 *
 * Author: John Thompson, Greenstone Project, NZDL, University of Waikato
 *
 * Copyright (C) 2003 New Zealand Digital Library Project
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *########################################################################
 */
package org.greenstone.gatherer.gui;

import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import org.greenstone.gatherer.Configuration;
import org.greenstone.gatherer.Dictionary;
import org.greenstone.gatherer.Gatherer;
import org.greenstone.gatherer.util.JarTools;


/** A dialog that warns about something, and allows the user to turn off the warning in the future. */
public class WarningDialog
    extends ModalDialog 
    implements ActionListener, KeyListener
{
    static final private Dimension NORMAL_SIZE = new Dimension(450, 160);
    static final private Dimension SETTING_SIZE = new Dimension(450, 200);

    private int result = JOptionPane.CANCEL_OPTION;
    private JButton cancel_button;
    private JButton ok_button;
    private JCheckBox show_check;
    private JComponent value_field;
    private JPanel value_panel;
    private String affected_property;
    private String full_property;
    /** Whether or not to display the checkbox that will turn off this dialog in future */
    private final boolean showcheckbox;


    public WarningDialog(String warning_name, String warning_title, String warning_message, String affected_property, boolean can_cancel) 
    {
	this(warning_name, warning_title, warning_message, affected_property, can_cancel, true); // true for show checkbox
    }


    public WarningDialog(String warning_name, String warning_title, String warning_message, String affected_property, boolean can_cancel,
			 boolean showcheckbox)
    {
	super(Gatherer.g_man, "Warning", true);
        this.setComponentOrientation(Dictionary.getOrientation());
	this.showcheckbox = showcheckbox; 

	// Determine the name of this prompt.
	this.affected_property = affected_property;
	this.full_property = warning_name;

	// Now build dialog.
	if (affected_property != null) {
	    setSize(SETTING_SIZE);
	}
	else {
	    setSize(NORMAL_SIZE);
	}
	setTitle(warning_title);
	
	// Creation
	JPanel content_pane = (JPanel) getContentPane();
        content_pane.setComponentOrientation(Dictionary.getOrientation());
	JPanel text_pane = new JPanel();
        text_pane.setComponentOrientation(Dictionary.getOrientation());
        
	String gmedium_image = "gatherer_medium.png";
	if (Configuration.fedora_info != null && Configuration.fedora_info.isActive()) {
	    gmedium_image = "fli-" + gmedium_image;
	}

	JLabel icon_label = new JLabel(JarTools.getImage(gmedium_image));
        icon_label.setComponentOrientation(Dictionary.getOrientation());

	JTextArea text_area = new JTextArea();
        text_area.setComponentOrientation(Dictionary.getOrientation());
	text_area.setEditable(false);
	text_area.setLineWrap(true);
	text_area.setText(warning_message);
	text_area.setCaretPosition(0);
	text_area.setWrapStyleWord(true);

	value_panel = new JPanel();
        value_panel.setComponentOrientation(Dictionary.getOrientation());
	JLabel value_label = new JLabel(Dictionary.get("WarningDialog.Value"));
	value_label.setComponentOrientation(Dictionary.getOrientation());
        
	value_field = new JTextField();
        value_field.setComponentOrientation(Dictionary.getOrientation());
	JPanel bottom_pane = new JPanel();
        bottom_pane.setComponentOrientation(Dictionary.getOrientation());
        
	JPanel control_pane = new JPanel();
        control_pane.setComponentOrientation(Dictionary.getOrientation());
	ok_button = new GLIButton(Dictionary.get("General.OK"), Dictionary.get("General.OK_Tooltip"));
	cancel_button = new GLIButton(Dictionary.get("General.Cancel"), Dictionary.get("General.Pure_Cancel_Tooltip"));
	
	// Connection
	ok_button.addActionListener(this);
	cancel_button.addActionListener(this);
	ok_button.addKeyListener(this);
	cancel_button.addKeyListener(this);
	getRootPane().setDefaultButton(ok_button);

	// Layout
	icon_label.setBorder(BorderFactory.createEmptyBorder(0,0,0,5));

	value_label.setBorder(BorderFactory.createEmptyBorder(0, icon_label.getPreferredSize().width, 0, 0));

	value_panel.setBorder(BorderFactory.createEmptyBorder(5,0,0,0));
	value_panel.setLayout(new BorderLayout(5,0));
	value_panel.add(value_label, BorderLayout.LINE_START);
	value_panel.add(value_field, BorderLayout.CENTER);

	text_pane.setLayout(new BorderLayout());
	text_pane.add(icon_label, BorderLayout.LINE_START);
	text_pane.add(new JScrollPane(text_area), BorderLayout.CENTER);
	if(affected_property != null) {
	    text_pane.add(value_panel, BorderLayout.SOUTH);
	}

	if(can_cancel) {
	    control_pane.setLayout(new GridLayout(1,2,5,0));
	    control_pane.add(ok_button);
	    control_pane.add(cancel_button);
	}
	else {
	    control_pane.setLayout(new BorderLayout());
	    control_pane.add(ok_button, BorderLayout.LINE_END);
	}

	bottom_pane.setBorder(BorderFactory.createEmptyBorder(5,0,0,0));
	bottom_pane.setLayout(new BorderLayout());
	bottom_pane.add(control_pane, BorderLayout.LINE_END);
	if(showcheckbox) {
	    show_check = new JCheckBox(Dictionary.get("WarningDialog.Dont_Show_Again"));
	    show_check.setComponentOrientation(Dictionary.getOrientation());
	    bottom_pane.add(show_check, BorderLayout.CENTER);
	}

	content_pane.setBorder(BorderFactory.createEmptyBorder(5,5,5,5));
	content_pane.setLayout(new BorderLayout());
	content_pane.add(text_pane, BorderLayout.CENTER);
	content_pane.add(bottom_pane, BorderLayout.SOUTH);

	// Position
	Dimension size = getSize();
	if (Gatherer.g_man != null) {
	    Rectangle frame_bounds = Gatherer.g_man.getBounds();
	    setLocation(frame_bounds.x + (frame_bounds.width - size.width) / 2, frame_bounds.y + (frame_bounds.height - size.height) / 2);
	}
	else {
	    Dimension screen_size = Toolkit.getDefaultToolkit().getScreenSize();
	    setLocation((screen_size.width - size.width) / 2, (screen_size.height - size.height) / 2);
	}
    }

    public void actionPerformed(ActionEvent event) {
	boolean valid_value = true;
	if(event.getSource() == ok_button) {
	    if(affected_property != null && Configuration.self != null) {
		valid_value = URLField.validateURL(value_field);
		if(valid_value) {
		    // Store the value of the property
		    URLField.store(value_field, affected_property); //Configuration.setString(affected_property, true, value);
		}
	    }
	    if(valid_value) {
		result = JOptionPane.OK_OPTION;
	    }
	} 

	if(valid_value) {
	    if (Configuration.self != null && showcheckbox) {
		// Store the state of the show message checkbox.
		Configuration.set(full_property, true, !show_check.isSelected());
	    }
	    // Done.
	    setVisible(false);
	}
	else {
	    JOptionPane.showMessageDialog(Gatherer.g_man, Dictionary.get("WarningDialog.Invalid_Value"), Dictionary.get("General.Error"), JOptionPane.ERROR_MESSAGE);
	}
    }

    /** Gives notification of key events on the buttons */
    public void keyPressed(KeyEvent e) {
	if (e.getKeyCode() == KeyEvent.VK_ENTER) {
	    // Enter: Click the button in focus
	    Object source = e.getSource();
	    if (source instanceof AbstractButton) {
		((AbstractButton) source).doClick();
	    }
	}
    }

    public void keyReleased(KeyEvent e) {}

    public void keyTyped(KeyEvent e) {}


    public int display() {
	if (Configuration.self == null || Configuration.get(full_property, false)) {
	    setVisible(true);
	}
	// We are no longer showing this dialog, so result must always be true.
	else {
	    result = JOptionPane.OK_OPTION;
	}
	// Ask the parent to repaint, just to be sure
	if(Gatherer.g_man != null) {
	    Gatherer.g_man.repaint();
	}
	return result;
    }

    /** Allows you to specify whether this dialog is a warning dialog, or only a message dialog.
     * @param message_only true if this dialog shows a message, false for a warning
     */
    public void setMessageOnly(boolean message_only) {
	if(!showcheckbox) {
	    return;
	}

	// If this is a message then change the checkbox
	if(message_only) {
	    show_check.setText(Dictionary.get("WarningDialog.Dont_Show_Again_Message"));
	}
	// And if its a warning, change them back
	else {
	    show_check.setText(Dictionary.get("WarningDialog.Dont_Show_Again"));
	}
    }

    /** Allows you to replace the generic text field control with a JTextField subclass with further functionality. For instance you might provide a URLField to allow only valid URLs to be accepted.
     * @param  control the JTextField subclass you want to use for the control
     */
    public void setValueField(JComponent control) {
	// Remove the current control
	value_panel.remove(value_field);
	// Replace with the new one
	value_field = control;
	// Re-add
	value_panel.add(value_field, BorderLayout.CENTER);
    }

    protected void processWindowEvent(WindowEvent event) {
	if(event.getID() == WindowEvent.WINDOW_ACTIVATED) {
	    if(affected_property != null) {
		value_field.requestFocus();
	    }
	    else {
		ok_button.requestFocus();
	    }
	}
	else {
	    super.processWindowEvent(event);
	}
    }

    public void setValueField(String field_value){
	URLField.setText(this.value_field, field_value);
    }

}
