/**
 *#########################################################################
 *
 * A component of the Gatherer application, part of the Greenstone digital
 * library suite from the New Zealand Digital Library Project at the
 * University of Waikato, New Zealand.
 *
 * Author: John Thompson, Greenstone Digital Library, University of Waikato
 *
 * Copyright (C) 1999 New Zealand Digital Library Project
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *########################################################################
 */
package org.greenstone.gatherer.cdm;

import java.util.*;
import org.greenstone.gatherer.util.StaticStrings;
import org.greenstone.gatherer.util.XMLTools;
import org.w3c.dom.*;

/** This class encapsulates a single indexing pair.
 * @author John Thompson, Greenstone Digital Library, University of Waikato
 * @version 2.4
 */
public class SubcollectionIndex
    implements Comparable, DOMProxyListEntry {

    private ArrayList sources = null;
    /** The element this index is based upon. */
    private Element element = null;
    /** The unique, if cryptic, identifier of an index. */
    private String id = null;

    /** Default constructor, which should only be used during DOMProxyListModel creation. */
    public SubcollectionIndex() {
    }

    /** Constructor. */
    public SubcollectionIndex(Element element) { 
	this.element = element;
    }

    public SubcollectionIndex(Object[] raw_sources) {
	this.sources = new ArrayList();
	// Create a new element
	element = CollectionConfiguration.createElement(StaticStrings.INDEX_ELEMENT);
	// For each source add a content element
	int size = raw_sources.length;
	for(int i = 0; i < size; i++) {
	    Subcollection subcollection = (Subcollection) raw_sources[i];
	    String subcollection_name = subcollection.getName();
	    sources.add(subcollection_name);
	    Element content_element = CollectionConfiguration.createElement(StaticStrings.CONTENT_ELEMENT);
	    content_element.setAttribute(StaticStrings.NAME_ATTRIBUTE, subcollection_name);
	    element.appendChild(content_element);
	    content_element = null;
	}
    }

    /** Method to compare two indexes.
     * @param object The other index as an <strong>Object</strong>.
     * @return An <i>int</i> which indicates how the indexes compare.
     * @see java.lang.String
     */
    public int compareTo(Object object) {
	return getID().compareTo(((SubcollectionIndex)object).getID());
    }
    
    public DOMProxyListEntry create(Element element) {      	
	return new SubcollectionIndex(element);
    }

    /** Method to test for the equality of two indexes.
     * @param object The other index as an <strong>Object</strong>.
     * @return A <i>boolean</i> which is <i>true</i> if the two indexes are equal, <i>false</i> otherwise.
     */
    public boolean equals(Object object) {
	return (compareTo(object) == 0);
    }
    
    public Element getElement() {
	return element;
    }

    public String getID() {
	if(id == null) {
	    StringBuffer id_buffer = new StringBuffer();
	    // Write data information. Retrieve each of the content sources and add them in a comma separated list.
	    ArrayList sources = getSources();
	    int sources_size = sources.size();
	    for(int i = 0; i < sources_size; i++) {
		id_buffer.append((String)sources.get(i));
		id_buffer.append(StaticStrings.COMMA_CHARACTER);
	    }
	    sources = null;
	    id = id_buffer.substring(0, id_buffer.length() - 1);
	}
	return id;
    }

    /** Retrieve the sources of this index.
     * @return the sources as an ArrayList
     */
    public ArrayList getSources() {
	if(sources == null) {
	    sources = new ArrayList();
	    // for the last entry in the assigned subcollection partitions, element is null
	    // why this is null for the last entry?
	    if(element==null){	 
	      return sources;
	    } 
	    NodeList content_elements = element.getElementsByTagName(StaticStrings.CONTENT_ELEMENT);	    
	    int content_elements_length = content_elements.getLength();
	    for(int i = 0; i < content_elements_length; i++) {
		Element content_element = (Element) content_elements.item(i);
		sources.add(content_element.getAttribute(StaticStrings.NAME_ATTRIBUTE));
	    }
	    content_elements = null;
	    Collections.sort(sources);
	}
	return sources;
    }

    public boolean isAssigned() {
	return (element != null && !element.getAttribute(StaticStrings.ASSIGNED_ATTRIBUTE).equals(StaticStrings.FALSE_STR));
    }

    public void setAssigned(boolean assigned) {
	if(element != null) {
	    element.setAttribute(StaticStrings.ASSIGNED_ATTRIBUTE, (assigned ? StaticStrings.TRUE_STR : StaticStrings.FALSE_STR));
	}
    }

    public void setElement(Element element) {
	this.element = element;
	this.id = null;
	this.sources = null;
    }

    /** Method to set the sources for this index which can only be used for the default index.
     * @param sources an ArrayList of source names
     */
    public void setSources(ArrayList sources) {
	if(element != null) {
	    // Erase old sources
	    XMLTools.clear(element);
	    // For each entry in the sources array add a new content element.
	    int size = sources.size();
	    for(int i = 0; i < size; i++) {
		Element content_element = element.getOwnerDocument().createElement(StaticStrings.CONTENT_ELEMENT);
		content_element.setAttribute(StaticStrings.NAME_ATTRIBUTE, (String) sources.get(i));
		element.appendChild(content_element);
		content_element = null;
	    }
	    this.id = null; // Regenerate ID.
	    this.sources = sources;
	}
    }

    /** Method to turn this object into a string representation ready to be placed in the collection configuration file.
     * @return A <strong>String</strong> containing the information of this class.
     */

    public String toString() {
	return getID();
    }
}
