/**
 *#########################################################################
 *
 * A component of the Gatherer application, part of the Greenstone digital
 * library suite from the New Zealand Digital Library Project at the
 * University of Waikato, New Zealand.
 *
 * <BR><BR>
 *
 * Author: John Thompson, Greenstone Digital Library, University of Waikato
 *
 * <BR><BR>
 *
 * Copyright (C) 1999 New Zealand Digital Library Project
 *
 * <BR><BR>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * <BR><BR>
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * <BR><BR>
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *########################################################################
 */
package org.greenstone.gatherer.gui;

import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import org.greenstone.gatherer.Configuration;
import org.greenstone.gatherer.Dictionary;
import org.greenstone.gatherer.Gatherer;

/** A class that extends a JDialog into a editor for editing large block of text for the metadata value.
 * @author John Thompson, Greenstone Digital Library, University of Waikato
 * @version 2.3
 */
final public class EditorDialog
    extends ModalDialog 
    implements ActionListener {
   /** Is this dialog editable? */
   private boolean editable = true;
   /** The cancel, and I don't want the text I've typed, button. */
    private JButton cancel = null;
    /** The ok, I'll save what I've just typed in, button. */
    private JButton ok = null;
    /** The area in which we type. */
    private JTextArea text = null;
    /** And what result should be passed back to our caller. */
    private String result = null;
    /** The size of the edit pop-up. */
    final static private Dimension SIZE = new Dimension(400, 300);

    /** Constructor */
    public EditorDialog() {
	super(Gatherer.g_man);
        this.setComponentOrientation(Dictionary.getOrientation());
    }

    /** Any implementation of ActionListener must include this method so we can be informed when an action has been performed on one of our target controls. In this case we generate a pop-up window to edit in.
     * @param event An <strong>ActionEvent</strong> containing information about the event.
     */
    public void actionPerformed(ActionEvent event) {
	if (event.getSource() == ok) {
	    result = text.getText();
	}
	dispose();
    }

    /** Method to display the editing box on screen.
     * @param value The initial text to be displayed in the editing area, as a <strong>String</strong>.
     * @return The new value for the metadata value as a <strong>String</strong> or <i>null</i> if the user has pressed cancel.
     */
    public String display(String value) {
	setModal(true);
	setSize(SIZE);
	setJMenuBar(new SimpleMenuBar("theenrichview"));
	if (editable) {
	    setTitle(Dictionary.get("General.Edit"));
	} else {
	    setTitle(Dictionary.get("General.View"));
	}
	// Create
	text = new JTextArea(value);
        text.setComponentOrientation(Dictionary.getOrientation());
	text.setCaretPosition(value.length());
	text.setEditable(editable);
	text.setLineWrap(true);
	text.setWrapStyleWord(false);
	if (editable) {
	    text.setToolTipText(Dictionary.get("EnrichPane.Value_Field_Tooltip"));
	} else {
	    text.setToolTipText(Dictionary.get("EnrichPane.Value_Field_Tooltip_Uneditable"));
	}

	cancel = new GLIButton(Dictionary.get("General.Cancel"), Dictionary.get("General.Pure_Cancel_Tooltip"));
	
	ok = new GLIButton(Dictionary.get("General.OK"), Dictionary.get("General.OK_Tooltip"));
	

	// Listeners
	cancel.addActionListener(this);
	ok.addActionListener(this);

	// Layout
	JPanel button_pane = new JPanel();
        button_pane.setComponentOrientation(Dictionary.getOrientation());
	button_pane.setLayout(new GridLayout(1,2));
	if(editable) {
	   button_pane.add(ok);
	   button_pane.add(cancel);
	}
	else {
	   button_pane.add(new JPanel());
	   button_pane.add(ok);
	}
		  
	JPanel content_pane = (JPanel) getContentPane();
        content_pane.setComponentOrientation(Dictionary.getOrientation());
	content_pane.setLayout(new BorderLayout());
        JScrollPane scrol_tmp;
        scrol_tmp = new JScrollPane(text);
        scrol_tmp.setComponentOrientation(Dictionary.getOrientation());
	content_pane.add(scrol_tmp, BorderLayout.CENTER);
	content_pane.add(button_pane, BorderLayout.SOUTH);
		  
	Dimension screen_size = Configuration.screen_size;
	setLocation((screen_size.width - SIZE.width) / 2, (screen_size.height - SIZE.height) / 2);
	setVisible(true);
	return result;
    }

    /** Specify if this text dialog should be editable or readonly
     * @param editable true to allow editing, false otherwise
     */
    public void setEditable(boolean editable) {
      if(text == null) {
	 this.editable = editable;
      }
      else {
	 text.setEditable(editable);
      }
   }
}
