/**********************************************************************
 *
 * OIDtools.cpp -- 
 * Copyright (C) 1999-2008  The New Zealand Digital Library Project
 *
 * A component of the Greenstone digital library software
 * from the New Zealand Digital Library Project at the
 * University of Waikato, New Zealand.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *********************************************************************/

#include "OIDtools.h"


// returns (in top) the top level of OID (i.e. everything
// up until the first dot)
void get_top (const text_t &OID, text_t &top)
{
  top.clear();
  if (OID.empty()) return;

  text_t::const_iterator begin = OID.begin();
  text_t::const_iterator end = OID.end();

  top.appendrange (begin, findchar(begin, end, '.'));
}	


// checks if OID is top level (i.e. contains no dots)
bool is_top (const text_t &OID)
{
  if (OID.empty()) return true;

  text_t::const_iterator here = OID.begin();
  text_t::const_iterator end = OID.end();
  here = findchar (here, end, '.');

  if (here == end) return true;
  return false;
}


// get_parents_array loads the parents array with all the parents of the
// document or classification specified by OID (not including OID itself)
void get_parents_array (const text_t &OID, text_tarray &parents)
{
  text_t::const_iterator here = OID.begin ();
  text_t::const_iterator end = OID.end ();
  text_t thisparent;

  while (here != end) {
    if (*here == '.') parents.push_back(thisparent);
    thisparent.push_back(*here);
    ++here;
  }
}


// get_parent returns the parent of the document or classification
// specified by OID
text_t get_parent (const text_t& OID)
{
  if (OID.empty() || is_top (OID)) return g_EmptyText;

  text_t::const_iterator begin = OID.begin();
  text_t::const_iterator here = (OID.end() - 1);

  while (here >= begin) {
    if (*here == '.')
      break;
    if (here == begin)
      break;
    --here;
  }

  if (here != begin) {
    text_t parentOID;
    parentOID.appendrange(begin, here);
    return parentOID;
  }

  return g_EmptyText;
}


// takes an OID like ".2 and replaces the " with parent
void translate_parent (text_t &OID, const text_t &parent)
{
  text_t::const_iterator here = OID.begin();
  text_t::const_iterator end = OID.end();
  text_t temp;

  while (here != end) {
    if (*here == '"') temp += parent;
    else temp.push_back (*here);
    ++here;
  }
  OID = temp;
}


// shrink_parent does the opposite to translate_parent
void shrink_parent (text_t &OID)
{
  text_tarray tmp;
  splitchar (OID.begin(), OID.end(), '.', tmp);
  OID = "\"." + tmp.back();
}


// checks if OID uses ".fc", ".lc", ".pr", "rt", ".ns",
// or ".ps" syntax (first child, last child, parent, root,
// next sibling, previous sibling)
bool needs_translating (const text_t &OID)
{
  if (OID.size() < 4) return false;

  text_t tail = substr (OID.end()-3, OID.end());
  if (tail == ".fc" || tail == ".lc" || tail == ".pr" || 
      tail == ".rt" || tail == ".ns" || tail == ".ps") return true;

  return false;
}


// strips the ".fc", ".lc", ".pr", ".ns",
// or ".ps" suffix from the end of OID
void strip_suffix (text_t &OID)
{
  text_t tail = substr (OID.end()-3, OID.end());
  while (tail == ".fc" || tail == ".lc" || tail == ".pr" || 
	 tail == ".rt" || tail == ".ns" || tail == ".ps") {
    OID.erase (OID.end()-3, OID.end());
    tail = substr (OID.end()-3, OID.end());
  }
}


// is_child_of returns true if OID2 is a child of OID1
bool is_child_of(const text_t &OID1, const text_t &OID2)
{
  text_t parent = get_parent(OID2);

  while (!parent.empty()) {
    if (parent == OID1) return true;
    parent = get_parent(parent);
  }

  return false;
}
