/**********************************************************************
 *
 * GS2LuceneQuery.java 
 *
 * Copyright 2004 The New Zealand Digital Library Project
 *
 * A component of the Greenstone digital library software
 * from the New Zealand Digital Library Project at the
 * University of Waikato, New Zealand.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *********************************************************************/
package org.greenstone.LuceneWrapper4;


import java.io.*;
import java.util.*;
import java.util.regex.*;

import org.apache.lucene.analysis.Analyzer;
import org.apache.lucene.analysis.standard.StandardAnalyzer;
import org.apache.lucene.document.Document;
import org.apache.lucene.index.DirectoryReader;
import org.apache.lucene.index.IndexReader;
import org.apache.lucene.index.Term;
//import org.apache.lucene.index.TermDocs;
import org.apache.lucene.queryparser.classic.ParseException;
import org.apache.lucene.queryparser.classic.QueryParser;
import org.apache.lucene.search.BooleanClause;
import org.apache.lucene.search.BooleanQuery; // for the TooManyClauses exception
import org.apache.lucene.search.ConstantScoreQuery;
import org.apache.lucene.search.Filter;
import org.apache.lucene.search.IndexSearcher; // Searcher is deprecated
import org.apache.lucene.search.MultiTermQuery;
import org.apache.lucene.search.MultiTermQuery.ConstantScoreAutoRewrite;
import org.apache.lucene.search.Query;
import org.apache.lucene.search.TermRangeFilter;
import org.apache.lucene.search.ScoreDoc;
import org.apache.lucene.search.Sort;
import org.apache.lucene.search.SortField;
import org.apache.lucene.search.TopFieldDocs;

import org.apache.lucene.index.DocsEnum;
import org.apache.lucene.index.MultiFields;

import org.apache.lucene.store.Directory;
import org.apache.lucene.store.FSDirectory;

import org.apache.lucene.util.Bits;
import org.apache.lucene.util.BytesRef;
import org.apache.lucene.util.Version;

public class GS2LuceneQuery extends SharedSoleneQuery
{
  public static String SORT_RANK = "rank";
  public static String SORT_NATURAL = "natural";

    protected String full_indexdir="";

  protected SortField.Type sort_type = SortField.Type.SCORE;
  protected boolean reverse_sort = false;
    protected Sort sorter=new Sort();
    protected Filter filter = null;

    protected QueryParser query_parser = null;
    protected QueryParser query_parser_no_stop_words = null;
    protected IndexSearcher searcher = null;
    protected IndexReader reader = null; // reference to a Reader resource. GS2LuceneQuery doesn't maintain it, GS2LuceneSearch maintains it!
		// GS2LuceneSearch locally instantiates one GS2LuceneQuery object per query then allows each Query instance use a relevant Reader.
		// But GS2LuceneSearch opens the IndexReaders and, more importantly, closes them all when a collection is deactivated.

    public GS2LuceneQuery() {
	super();

	// Create one query parser with the standard set of stop words, and one with none

	query_parser = new QueryParser(GSLuceneConstants.MATCH_VERSION, TEXTFIELD, new GS2Analyzer()); // uses built-in stop_words_set
    	query_parser_no_stop_words = new QueryParser(GSLuceneConstants.MATCH_VERSION, TEXTFIELD, new GS2Analyzer(new String[] { }));
    }   
	
    public boolean initialise(IndexReader reader) {

		if (!super.initialise()) {
			return false;
		}


    	if (full_indexdir==null || full_indexdir.length()==-1){
	    utf8out.println("Index directory is not indicated ");
	    utf8out.flush();
	    return false;
    	}
		
		if(reader == null) {
			return false;
		}
		else {
			this.reader = reader;
			this.searcher = new IndexSearcher(reader); // during searcher.search() will get it to compute ranks when sorting by fields
			this.sorter = new Sort(new SortField(this.sort_field, this.sort_type, this.reverse_sort));		
			return true;
		}
    }

    public void setIndexDir(String full_indexdir) {
	this.full_indexdir = full_indexdir;
    }

    public void setSortField(String sort_field) {
      if (sort_field.equals(SORT_RANK)) {
	this.sort_field = null;
	this.sort_type = SortField.Type.SCORE;
      } else if (sort_field.equals(SORT_NATURAL)) {
	this.sort_field = null;
	this.sort_type = SortField.Type.DOC;
      } else {
	this.sort_field  = sort_field;
	this.sort_type = SortField.Type.STRING; // for now. numeric??
      }
    }
  public void setReverseSort(boolean reverse) {
    this.reverse_sort = reverse;
  }
  public boolean getReverseSort() {
    return this.reverse_sort;
  }

    public void setFilterString(String filter_string) {
	super.setFilterString(filter_string);
	this.filter = parseFilterString(filter_string);
    }

    public Filter getFilter() {
	return this.filter;
    }

    
    public LuceneQueryResult runQuery(String query_string) {
	
	if (query_string == null || query_string.equals("")) {
	    utf8out.println("The query word is not indicated ");
	    utf8out.flush();
	    return null;
	}

 	LuceneQueryResult lucene_query_result=new LuceneQueryResult();
	lucene_query_result.clear();
	
	if(this.reader == null) {
		System.err.println("#### Reader is null!");
	}
	
	try {    	        
	    Query query_including_stop_words = query_parser_no_stop_words.parse(query_string);
	    query_including_stop_words = query_including_stop_words.rewrite(reader);
		
	    // System.err.println("********* query_string " + query_string + "****");

	    Query query = parseQuery(reader, query_parser, query_string, fuzziness);
	    query = recursivelyRewriteQuery(query, reader, lucene_query_result);	    
	    // System.err.println("@@@@ final query class name: " + query.getClass());

	    // http://stackoverflow.com/questions/13537126/term-frequency-in-lucene-4-0
	    // http://stackoverflow.com/questions/20575254/lucene-4-4-how-to-get-term-frequency-over-all-index
	    // http://stackoverflow.com/questions/8938960/how-to-get-document-ids-for-document-term-vector-in-lucene?rq=1
	    // https://github.com/hibernate/hibernate-search/blob/master/orm/src/test/java/org/hibernate/search/test/filter/BestDriversFilter.java
	    // http://lucene.apache.org/core/4_7_2/MIGRATE.html

	    // Get the list of expanded query terms and their frequencies 
	    // num docs matching, and total frequency	    
	    HashSet terms = new HashSet();
	    query.extractTerms(terms);

	    HashMap doc_term_freq_map = new HashMap();
	    
	    Iterator iter = terms.iterator();
	    
	    Bits liveDocs = null;
	    if(reader.hasDeletions()) {
		System.err.println("@@@ GS2LuceneQuery.java: There have been deletions in index at "+this.full_indexdir +". Merging to get liveDocs.");
		liveDocs = MultiFields.getLiveDocs(reader); // SLOW! But getLiveDocs returns null if there are no deletions
	    }

	    while (iter.hasNext()) {

		// http://stackoverflow.com/questions/13537126/term-frequency-in-lucene-4-0
		    
		Term term = (Term) iter.next();
		// System.err.println("@@@ GS2LuceneQuery.java: Next term: " + term.text());
		BytesRef term_bytes = term.bytes();
		DocsEnum term_docs = MultiFields.getTermDocsEnum(reader, liveDocs, term.field(), term_bytes); // flags?

		// Get the term frequency over all the documents
		//TermDocs term_docs = reader.termDocs(term);
		int term_freq = 0;
		int match_docs = 0;

		if(term_docs != null) {
		    int docID = -1;
		    while((docID = term_docs.nextDoc()) != DocsEnum.NO_MORE_DOCS) {//while (term_docs.next())	
			if (term_docs.freq() != 0)
			    {
				term_freq += term_docs.freq();
				match_docs++;
				
				// Calculate the document-level term frequency as well
				Integer lucene_doc_num_obj = new Integer(term_docs.docID());
				int doc_term_freq = 0;
				if (doc_term_freq_map.containsKey(lucene_doc_num_obj))
				    {
					doc_term_freq = ((Integer) doc_term_freq_map.get(lucene_doc_num_obj)).intValue();
				    }
				doc_term_freq += term_docs.freq();
				
				doc_term_freq_map.put(lucene_doc_num_obj, new Integer(doc_term_freq));
			    }
		    }
		}
                // else {
		//     System.err.println("@@@ GS2LuceneQuery.java: term_docs is null for term " + term.text());
		// }

		// Create a term 
		lucene_query_result.addTerm(term.text(), term.field(), match_docs, term_freq);
	    }
	
	    // Get the list of stop words removed from the query
	    HashSet terms_including_stop_words = new HashSet();
	    query_including_stop_words.extractTerms(terms_including_stop_words);
	    Iterator terms_including_stop_words_iter = terms_including_stop_words.iterator();
	    while (terms_including_stop_words_iter.hasNext()) {
		Term term = (Term) terms_including_stop_words_iter.next();
		if (!terms.contains(term)) {
		    lucene_query_result.addStopWord(term.text());
		}
	    }
	    
	    // Extracting all documents for a given search - http://www.gossamer-threads.com/lists/lucene/java-user/134873
	    // http://lucene.apache.org/core/3_4_0/api/core/org/apache/lucene/search/TotalHitCountCollector.html
	    // http://lucene.apache.org/core/4_7_2/core/index.html?org/apache/lucene/search/TopFieldDocs.html

	    // 1. Figure out how many results there will be.
	    //TotalHitCountCollecter countCollector = new TotalHitCountCollector();
	    //searcher.search(query, filter, collector);
	    //int hitCount = collector.count;

	    // Actually do the query
	    // Simple case for getting all the matching documents
	    if (end_results == Integer.MAX_VALUE) {
		// Perform the query (filter and sorter may be null)
		TopFieldDocs hits = searcher.search(query, filter, end_results, sorter, true, true); // doDocScores=true, doMaxScore=true
		     // Is there a slight difference in the definition between
		     // https://lucene.apache.org/core/3_6_0/api/all/org/apache/lucene/search/IndexSearcher.html#setDefaultFieldSortScoring%28boolean,%20boolean%29
		     // and http://lucene.apache.org/core/4_7_2/core/org/apache/lucene/search/IndexSearcher.html#search%28org.apache.lucene.search.Query,%20org.apache.lucene.search.Filter,%20int,%20org.apache.lucene.search.Sort,%20boolean,%20boolean%29
		     // Seems to be okay.
		     // See also http://stackoverflow.com/questions/7910241/in-lucene-what-is-the-purpose-of-setdefaultfieldsortscoring

		lucene_query_result.setTotalDocs(hits.totalHits);

		// Output the matching documents
		lucene_query_result.setStartResults(start_results);
		lucene_query_result.setEndResults(hits.totalHits); // ??

		for (int i = start_results; i < hits.totalHits; i++) {
		  int lucene_doc_num = hits.scoreDocs[i ].doc; // i-1
		    Document doc = reader.document(lucene_doc_num);
		    int doc_term_freq = 0;
		    Integer doc_term_freq_object = (Integer) doc_term_freq_map.get(new Integer(lucene_doc_num));
		    if (doc_term_freq_object != null)
		    {
			doc_term_freq = doc_term_freq_object.intValue();
		    }
		    lucene_query_result.addDoc(doc.get("docOID").trim(), hits.scoreDocs[i].score, doc_term_freq);
		}
	    }

	    // Slightly more complicated case for returning a subset of the matching documents
	    else {
		// Perform the query (filter may be null)
		TopFieldDocs hits = searcher.search(query, filter, end_results, sorter, true, true); // doDocScores=true, doMaxScore=true
		       // See also http://stackoverflow.com/questions/7910241/in-lucene-what-is-the-purpose-of-setdefaultfieldsortscoring
		lucene_query_result.setTotalDocs(hits.totalHits);
		
		lucene_query_result.setStartResults(start_results);
		lucene_query_result.setEndResults(end_results < hits.scoreDocs.length ? end_results: hits.scoreDocs.length);

		// Output the matching documents
		for (int i = start_results; (i < hits.scoreDocs.length && i < end_results); i++) {
		    int lucene_doc_num = hits.scoreDocs[i].doc;
		    Document doc = reader.document(lucene_doc_num);
		    int doc_term_freq = 0;
		    Integer doc_term_freq_object = (Integer) doc_term_freq_map.get(new Integer(lucene_doc_num));
		    if (doc_term_freq_object != null)
		    {
			doc_term_freq = doc_term_freq_object.intValue();
		    }
		    lucene_query_result.addDoc(doc.get("docOID").trim(), hits.scoreDocs[i].score, doc_term_freq);
		}
	    }
	}
	
	catch (ParseException parse_exception) {
	    lucene_query_result.setError(LuceneQueryResult.PARSE_ERROR);
	}
	catch (BooleanQuery.TooManyClauses too_many_clauses_exception) {
	    lucene_query_result.setError(LuceneQueryResult.TOO_MANY_CLAUSES_ERROR);
	}
	catch (IOException exception) {
	    lucene_query_result.setError(LuceneQueryResult.IO_ERROR);
	    exception.printStackTrace();
	}
	catch (Exception exception) {
	    lucene_query_result.setError(LuceneQueryResult.OTHER_ERROR);
	    exception.printStackTrace();
	}
	return lucene_query_result;
    }

    public void setDefaultConjunctionOperator(String default_conjunction_operator) {
	super.setDefaultConjunctionOperator(default_conjunction_operator);

	if (default_conjunction_operator.equals("AND")) {
	    query_parser.setDefaultOperator(query_parser.AND_OPERATOR);
	    query_parser_no_stop_words.setDefaultOperator(query_parser.AND_OPERATOR);
	} else { // default is OR
	    query_parser.setDefaultOperator(query_parser.OR_OPERATOR);
	    query_parser_no_stop_words.setDefaultOperator(query_parser.OR_OPERATOR);
	}
    }
     
    // This version of the cleanUp() method is just to clean up anything associated only with this instance of GS2LuceneQuery.
	// So it won't clean up the singleton IndexReader instances maintained by the encapsulating GS2LuceneSearch class.
    public void cleanUp() {
		super.cleanUp();
		
		searcher = null;
	
		// Don't close the indexReader reference here.
		// This has moved into the GS2LuceneSearch.cleanUp() method, as it maintains singleton IndexReaders
		// for each index level (sidx, didix) with lifespans matching their collection's lifespan
		// A collection's GS2LuceneSearch object lives for the duration of the Collection.
		// A GS2LuceneQuery object is ephemeral: only lives for the duration of a query, allowing multiple
		// users to do queries concurrently, sharing a single IndexReader object for each indexing level
		// since IndexReaders support concurrency.
    }

    protected Query parseQuery(IndexReader reader, QueryParser query_parser, String query_string, String fuzziness)
	throws java.io.IOException, org.apache.lucene.queryparser.classic.ParseException
    {
	// Split query string into the search terms and the filter terms
	// * The first +(...) term contains the search terms so count
	//   up '(' and stop when we finish matching ')'
	int offset = 0;
	int paren_count = 0;
	boolean seen_paren = false;
	while (offset < query_string.length() && (!seen_paren || paren_count > 0)) {
	    if (query_string.charAt(offset) == '(') {
		paren_count++;
		seen_paren = true;
	    }
	    if (query_string.charAt(offset) == ')') {
		paren_count--;
	    }
	    offset++;
	}
	String query_prefix = query_string.substring(0, offset);
	String query_suffix = query_string.substring(offset);
	
	///ystem.err.println("Prefix: " + query_prefix);
	///ystem.err.println("Suffix: " + query_suffix);
	
	Query query = query_parser.parse(query_prefix);
	query = query.rewrite(reader);
	
	// If this is a fuzzy search, then we need to add the fuzzy
	// flag to each of the query terms
	if (fuzziness != null && query.toString().length() > 0) {
	    
	    // Revert the query to a string
	    System.err.println("Rewritten query: " + query.toString());
	    // Search through the string for TX:<term> query terms
	    // and append the ~ operator. Note that this search will
	    // not change phrase searches (TX:"<term> <term>") as
	    // fuzzy searching is not possible for these entries.
	    // Yahoo! Time for a state machine!
	    StringBuffer mutable_query_string = new StringBuffer(query.toString());
	    int o = 0; // Offset
	    // 0 = BASE, 1 = SEEN_T, 2 = SEEN_TX, 3 = SEEN_TX:
	    int s = 0; // State
	    while(o < mutable_query_string.length()) {
		char c = mutable_query_string.charAt(o);
		if (s == 0 && c == TEXTFIELD.charAt(0)) {
		    ///ystem.err.println("Found T!");
		    s = 1;
		}
		else if (s == 1) {
		    if (c == TEXTFIELD.charAt(1)) {
			///ystem.err.println("Found X!");
			s = 2;
		    }
		    else {
			s = 0; // Reset
		    }
		}
		else if (s == 2) {
		    if (c == ':') {
			///ystem.err.println("Found TX:!");
			s = 3;
		    }
		    else {
			s = 0; // Reset
		    }
		}
		else if (s == 3) {
		    // Don't process phrases
		    if (c == '"') {
			///ystem.err.println("Stupid phrase...");
			s = 0; // Reset
		    }
		    // Found the end of the term... add the
		    // fuzzy search indicator
		    // Nor outside the scope of parentheses
		    else if (Character.isWhitespace(c) || c == ')') {
			///ystem.err.println("Yahoo! Found fuzzy term.");
			mutable_query_string.insert(o, '~' + fuzziness);
			o++;
			s = 0; // Reset
		    }
		}
		o++;
	    }
	    // If we were in the state of looking for the end of a
	    // term - then we just found it!
	    if (s == 3) {
		    
		mutable_query_string.append('~' + fuzziness);
	    }
	    // Reparse the query
	    ///ystem.err.println("Fuzzy query: " + mutable_query_string.toString() + query_suffix);
	    query = query_parser.parse(mutable_query_string.toString() + query_suffix);
	}
	else {
	    query = query_parser.parse(query_prefix + query_suffix);
	}
	
	return query;
    }

    // If you're dealing with a BooleanQuery, they need to be recursively rewritten
    // as they can contain queries with wildcards (WildcardQuery|PrefixQuery subclasses of MultiTermQuery)
    // e.g. season* farm
    // If MultiTermQuery, then expand here. e.g. WildcardQuerys like season*.
    // DON'T call this method from inside parseQuery() (in place of its query.rewrite()), because then wildcard
    // queries like season* won't contain Terms (extractTerms() will be empty) since the ConstantScoreQuerys
    // that a WildcardQuery gets rewritten to here will contain Filters in place of Terms.
    // Call this method from runQuery() after it calls parseQuery().
    // Now searches like these will work
    //    season* farm
    //    season* farm*
    // and not just searches like the following which already used to work:
    //    season*
    //    snail farm
    // Idea for the solution of recursively processing a BooleanQuery came from inspecting source code to BooleanQuery.java
    //    https://github.com/apache/lucene-solr/blob/master/lucene/core/src/java/org/apache/lucene/search/BooleanQuery.java
    // which also does a recursive rewrite. Unfortunately, the existing BooleanQuery does not handle MultiTermQuery
    // subcomponents.
    protected Query recursivelyRewriteQuery(Query orig_query, IndexReader reader, LuceneQueryResult lucene_query_result) throws java.io.IOException
    {
	//Query query = orig_query.rewrite(reader);
	Query query = orig_query;

	if(orig_query instanceof BooleanQuery) {
	    BooleanQuery booleanQuery = (BooleanQuery)orig_query;
	    List<BooleanClause> clauses = booleanQuery.clauses(); 
	    for (BooleanClause clause : clauses) {
		Query subQuery = clause.getQuery();
		subQuery = recursivelyRewriteQuery(subQuery, reader, lucene_query_result);
		clause.setQuery(subQuery);
	    }
	}
	
	// GS2's LuceneWrapper uses lucene-2.3.2. GS3's LuceneWrapper3 works with lucene-3.3.0. 
	    // This change in lucene core library for GS3 (present since after version 2.4.1) had the
	    // side-effect that searching on "econom*" didn't display what terms it was searching for, 
	    // whereas it had done so in GS2. 

	    // The details of this problem and its current solution are explained in the ticket 
	    // http://trac.greenstone.org/ticket/845

	    // We need to change the settings for the rewriteMethod in order to get searches on wildcards
	    // to produce search terms again when the query gets rewritten.

	    // We try, in order:
	    // 1. RewriteMethod set to BooleanQuery, to get it working as in GS2 which uses lucene-2.3.2
	    // it will expand wildcard searches to its terms when searching at both section AND doc level.
	    // If that throws a TooManyClauses exception (like when searching for "a*" over lucene demo collection)
	    // 2. Then try a custom rewriteMethod which sets termCountCutoff=350 and docCountPercent cutoff=0.1%
	    // If that throws a TooManyClauses exception (could perhaps happen if the collection has a huge number of docs
	    // 3. Then try the default apache rewriteMethod with its optimum defaults of 
	    // termCountCutoff=350 and docCountPercent cutoff=0.1%
	    // 	See http://lucene.apache.org/core/3_6_1/api/core/org/apache/lucene/search/MultiTermQuery.html

	    //System.err.println("@@@@ query class name: " + orig_query.getClass());
	    //System.err.println("@@@@ QUERY: " + orig_query);

	    if(orig_query instanceof MultiTermQuery) {
		MultiTermQuery multiTermQuery = (MultiTermQuery)orig_query;
		multiTermQuery.setRewriteMethod(MultiTermQuery.CONSTANT_SCORE_BOOLEAN_QUERY_REWRITE);
		     // less CPU intensive than MultiTermQuery.SCORING_BOOLEAN_QUERY_REWRITE)
	    }

	    try {
		query = orig_query.rewrite(reader);
	    } 
	    catch(BooleanQuery.TooManyClauses clauseException) {
		// Example test case: try searching the lucene demo collection for "a*" 
		// and you'll hit this exception

		lucene_query_result.setError(LuceneQueryResult.TOO_MANY_CLAUSES_ERROR);

		if(query instanceof MultiTermQuery) {

		    // CustomRewriteMethod: setting the docCountPercent cutoff to a custom 100%. 
		    // This will at least expand the query to its terms when searching with wildcards at section-level 
		    // (though it doesn't seem to work for doc-level searches, no matter what the cutoffs are set to).

		    MultiTermQuery.ConstantScoreAutoRewrite customRewriteMethod = new MultiTermQuery.ConstantScoreAutoRewrite();
		    customRewriteMethod.setDocCountPercent(100.0);
		    customRewriteMethod.setTermCountCutoff(350); // same as default
		    
		    MultiTermQuery multiTermQuery = (MultiTermQuery)query;
		    multiTermQuery.setRewriteMethod(customRewriteMethod);
		    try {
			query = query.rewrite(reader);
		    } 
		    catch(BooleanQuery.TooManyClauses clauseExceptionAgain) {

			// do what the code originally did: use the default rewriteMethod which
			// uses a default docCountPercent=0.1 (%) and termCountCutoff=350

			multiTermQuery.setRewriteMethod(MultiTermQuery.CONSTANT_SCORE_AUTO_REWRITE_DEFAULT);
			query = query.rewrite(reader);
		    }
		}
	    }

	    // BooleanQuery.java recurses rewriting any query until it is identical before and after rewrite,
	    // see reference to "recursively rewrite" in
	    // https://github.com/apache/lucene-solr/blob/master/lucene/core/src/java/org/apache/lucene/search/BooleanQuery.java
	    if(orig_query == query) {
		return query;
	    } else {
		return recursivelyRewriteQuery(query, reader, lucene_query_result);
	    }
    }

    protected Filter parseFilterString(String filter_string)
    {
	Filter result = null;
	Pattern pattern = Pattern.compile("\\s*\\+(\\w+)\\:([\\{\\[])(\\d+)\\s+TO\\s+(\\d+)([\\}\\]])\\s*");
	Matcher matcher = pattern.matcher(filter_string);
	if (matcher.matches()) {
	    String field_name = matcher.group(1);
	    boolean include_lower = matcher.group(2).equals("[");
	    BytesRef lower_term = new BytesRef(matcher.group(3));
	    BytesRef upper_term = new BytesRef(matcher.group(4));
	    boolean include_upper = matcher.group(5).equals("]");
	    result = new TermRangeFilter(field_name, lower_term, upper_term, include_lower, include_upper);
	}
	else {
	    System.err.println("Error: Could not understand filter string \"" + filter_string + "\"");
	}
	return result;
    }
    

    /** command line program and auxiliary methods */

    // Fairly self-explanatory I should hope
    static protected boolean query_result_caching_enabled = false;

    /**
     * This main() method is used by GS2 to do searches.
     * In GS2, lucene_query.pl calles this main() method in the LuceneWrapper4.jar. This main method instantiates both
     * a GS2LuceneQuery and an IndexReader object. It then passes the reader to the GS2LuceneQuery object by calling
     * the GS2LuceneQuery.initialise(reader) method. This main() method then finally performs the search with the provided query.
     * GS3 doesn't use this main() method. Instead a GS2LuceneSearch object (of gsdl3.jar) instantiates both
     * the GS2LuceneQuery and IndexReader objects and proceeds the same way.
     */
    static public void main (String args[])
    {
	if (args.length == 0) {
	    System.out.println("Usage: org.greenstone.LuceneWrapper4.GS2LuceneQuery <index directory> [-fuzziness value] [-filter filter_string] [-sort sort_field] [-reverse_sort][-dco AND|OR] [-startresults number -endresults number] [query]");
	    return;
	}

	try {
	    String index_directory = args[0];
	    
	    GS2LuceneQuery queryer = new GS2LuceneQuery();
	    queryer.setIndexDir(index_directory);

	    // Prepare the index cache directory, if query result caching is enabled
	    if (query_result_caching_enabled) {
		// Make the index cache directory if it doesn't already exist
		File index_cache_directory = new File(index_directory, "cache");
		if (!index_cache_directory.exists()) {
		    index_cache_directory.mkdir();
		}

		// Disable caching if the index cache directory isn't available
		if (!index_cache_directory.exists() || !index_cache_directory.isDirectory()) {
		    query_result_caching_enabled = false;
		}
	    }

	    String query_string = null;

	    // Parse the command-line arguments
            for (int i = 1; i < args.length; i++) {
		if (args[i].equals("-sort")) {
		    i++;
		    queryer.setSortField(args[i]);
		}
		else if (args[i].equals("-reverse_sort")) {
		  queryer.setReverseSort(true);
		}
		else if (args[i].equals("-filter")) {
		    i++;
		    queryer.setFilterString(args[i]);
		}
		else if (args[i].equals("-dco")) {
		    i++;
		    queryer.setDefaultConjunctionOperator(args[i]);
		}
		else if (args[i].equals("-fuzziness")) {
		    i++;
		    queryer.setFuzziness(args[i]);
		}
		else if (args[i].equals("-startresults")) {
		    i++;
		    if (args[i].matches("\\d+")) {
			queryer.setStartResults(Integer.parseInt(args[i]));
		    }
		}
		else if (args[i].equals("-endresults")) {
		    i++;
		    if (args[i].matches("\\d+")) {
			queryer.setEndResults(Integer.parseInt(args[i]));
		    }
		}
		else {
		    query_string = args[i];
		}
	    }
	    
	    Directory full_indexdir_dir = FSDirectory.open(new File(index_directory));
	    IndexReader reader = DirectoryReader.open(full_indexdir_dir); // Returns a IndexReader reading the index in the given Directory.
	                    // Now readOnly=true by default, and therefore also for searcher created in initialise() call below.
	    if (!queryer.initialise(reader)) {
		if(reader != null) reader.close(); // close reader object IF reader was instantiated
		queryer.cleanUp(); // will close searcher object if non-null
		return;
	    }
	    
	    // The query string has been specified as a command-line argument
	    if (query_string != null) {
		runQueryCaching(index_directory, queryer, query_string);
	    }

	    // Read queries from STDIN
	    else {
		BufferedReader in = new BufferedReader(new InputStreamReader(System.in, "UTF-8"));
		while (true) {
		    // Read the query from STDIN
		    query_string = in.readLine();
		    if (query_string == null || query_string.length() == -1) {
			break;
		    }

		    runQueryCaching(index_directory, queryer, query_string);
		    
		}
	    }
	    if(reader != null) reader.close();
	    queryer.cleanUp();
	}
 	catch (IOException exception) {
 	    exception.printStackTrace();
 	}
    }

    protected static void runQueryCaching(String index_directory, GS2LuceneQuery queryer, String query_string) 
	throws IOException
    {
	StringBuffer query_results_xml = new StringBuffer();

	// Check if this query result has been cached from a previous search (if it's enabled)
	File query_result_cache_file = null;
	if (query_result_caching_enabled) {
	    // Generate the cache file name from the query options
	    String query_result_cache_file_name = query_string + "-";
	    String fuzziness = queryer.getFuzziness();
	    query_result_cache_file_name += ((fuzziness != null) ? fuzziness : "") + "-";
	    String filter_string = queryer.getFilterString();
	    query_result_cache_file_name += ((filter_string != null) ? filter_string : "") + "-";
	    String sort_string = queryer.getSortField();
	    query_result_cache_file_name += ((sort_string != null) ? sort_string : "") + "-";
	    String reverse_sort_string = (queryer.getReverseSort() ? "1" : "0");
	    query_result_cache_file_name += reverse_sort_string + "-";
	    String default_conjunction_operator = queryer.getDefaultConjunctionOperator();
	    query_result_cache_file_name += default_conjunction_operator + "-";
	    int start_results = queryer.getStartResults();
	    int end_results = queryer.getEndResults();
	    query_result_cache_file_name += start_results + "-" + end_results;
	    query_result_cache_file_name = fileSafe(query_result_cache_file_name);

	    // If the query result cache file exists, just return its contents and we're done
	    File index_cache_directory = new File(index_directory, "cache");
	    query_result_cache_file = new File(index_cache_directory, query_result_cache_file_name);
	    if (query_result_cache_file.exists() && query_result_cache_file.isFile()) {
		FileInputStream fis = new FileInputStream(query_result_cache_file);
		InputStreamReader isr = new InputStreamReader(fis, "UTF-8");
		BufferedReader buffered_reader = new BufferedReader(isr);
		String line = "";
		while ((line = buffered_reader.readLine()) != null) {
		    query_results_xml.append(line + "\n");
		}
		String query_results_xml_string = query_results_xml.toString();
		query_results_xml_string = query_results_xml_string.replaceFirst("cached=\"false\"", "cached=\"true\"");

		utf8out.print(query_results_xml_string);
		utf8out.flush();

		return;
	    }
	}
	
	// not cached
	query_results_xml.append("<ResultSet cached=\"false\">\n");
	query_results_xml.append("<QueryString>" + LuceneQueryResult.xmlSafe(query_string) + "</QueryString>\n");
	Filter filter = queryer.getFilter();
	if (filter != null) {
	    query_results_xml.append("<FilterString>" + filter.toString() + "</FilterString>\n");
	}
	
	LuceneQueryResult query_result = queryer.runQuery(query_string);
	if (query_result == null) {
	    System.err.println("Couldn't run the query");
	    return;
	}
	
	if (query_result.getError() != LuceneQueryResult.NO_ERROR) {
	    query_results_xml.append("<Error type=\""+query_result.getErrorString()+"\" />\n");
	} else {
	    query_results_xml.append(query_result.getXMLString());
	}
	query_results_xml.append("</ResultSet>\n");

	utf8out.print(query_results_xml);
	utf8out.flush();

	// Cache this query result, if desired
	if (query_result_caching_enabled) {
	    // Catch any exceptions thrown trying to write the query result cache file and warn about them, but don't
	    //   bother with the full stack trace. It won't affect the functionality if we can't write some cache
	    //   files, it will just affect the speed of subsequent requests.
	    // Example exceptions are "permission denied" errors, or "filename too long" errors (the filter string
	    //   can get very long in some collections)
	    try
	    {
		FileWriter query_result_cache_file_writer = new FileWriter(query_result_cache_file);
		query_result_cache_file_writer.write(query_results_xml.toString());
		query_result_cache_file_writer.close();
	    }
	    catch (Exception exception)
	    {
		System.err.println("Warning: Exception occurred trying to write query result cache file (" + exception + ")");
	    }
	}
    }
    
    protected static String fileSafe(String text)
    {
	StringBuffer file_safe_text = new StringBuffer();
	for (int i = 0; i < text.length(); i++) {
	    char character = text.charAt(i);
	    if ((character >= 'A' && character <= 'Z') || (character >= 'a' && character <= 'z') || (character >= '0' && character <= '9') || character == '-') {
		file_safe_text.append(character);
	    }
	    else {
		file_safe_text.append('%');
		file_safe_text.append((int) character);
	    }
	}
	return file_safe_text.toString();
    }

    
}


