/**********************************************************************
 *
 * basequeryaction.cpp -- 
 * Copyright (C) 1999  The New Zealand Digital Library Project
 *
 * A component of the Greenstone digital library software
 * from the New Zealand Digital Library Project at the
 * University of Waikato, New Zealand.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *********************************************************************/

#include "basequeryaction.h"
#include "querytools.h"
#include "formattools.h"
#include "cgiutils.h"
#include "OIDtools.h"
#include "fileutil.h"
#include "text_t.h"
#include "historydb.h"
#include "htmlutils.h" // for html_safe in do_action
#include "gsdltools.h"
#include "phrases.h" // for get_phrases
#include <stdlib.h> // for strtol
#include <assert.h>

void colinfo_t::clear () {
  formatlistptr = NULL;
  browserptr = NULL;
}

void QueryResult_t::clear() {
  doc.clear();
  collection.clear();
}

basequeryaction::basequeryaction () {

  recpt = NULL;

  cgiarginfo arg_ainfo;

  // "q"
  arg_ainfo.shortname = "q";
  arg_ainfo.longname = "query string";
  arg_ainfo.multiplechar = true;
  arg_ainfo.multiplevalue = false;
  arg_ainfo.defaultstatus = cgiarginfo::weak;
  arg_ainfo.argdefault = g_EmptyText;
  arg_ainfo.savedarginfo = cgiarginfo::must;
  argsinfo.addarginfo (NULL, arg_ainfo);

  // "q2"
  arg_ainfo.shortname = "q2";
  arg_ainfo.longname = "query string for second query";
  arg_ainfo.multiplechar = true;
  arg_ainfo.multiplevalue = false;
  arg_ainfo.defaultstatus = cgiarginfo::weak;
  arg_ainfo.argdefault = g_EmptyText;
  arg_ainfo.savedarginfo = cgiarginfo::must;
  argsinfo.addarginfo (NULL, arg_ainfo);

  // "cq2" ""=don't combine, "and", "or", "not"
  arg_ainfo.shortname = "cq2";
  arg_ainfo.longname = "combine queries";
  arg_ainfo.multiplechar = true;
  arg_ainfo.multiplevalue = false;
  arg_ainfo.defaultstatus = cgiarginfo::weak;
  arg_ainfo.argdefault = g_EmptyText;
  arg_ainfo.savedarginfo = cgiarginfo::must;
  argsinfo.addarginfo (NULL, arg_ainfo);

 
  // "m"
  arg_ainfo.shortname = "m";
  arg_ainfo.longname = "maximum number of documents";
  arg_ainfo.multiplechar = true;
  arg_ainfo.multiplevalue = false;
  arg_ainfo.defaultstatus = cgiarginfo::weak;
  arg_ainfo.argdefault = "50";
  arg_ainfo.savedarginfo = cgiarginfo::must;
  argsinfo.addarginfo (NULL, arg_ainfo);

  // "o"
  arg_ainfo.shortname = "o";
  arg_ainfo.longname = "hits per page";
  arg_ainfo.multiplechar = true;
  arg_ainfo.multiplevalue = false;
  arg_ainfo.defaultstatus = cgiarginfo::weak;
  arg_ainfo.argdefault = "20";
  arg_ainfo.savedarginfo = cgiarginfo::must;
  argsinfo.addarginfo (NULL, arg_ainfo);

  // "r"
  arg_ainfo.shortname = "r";
  arg_ainfo.longname = "start results from";
  arg_ainfo.multiplechar = true;
  arg_ainfo.multiplevalue = false;
  arg_ainfo.defaultstatus = cgiarginfo::weak;
  arg_ainfo.argdefault = "1";
  arg_ainfo.savedarginfo = cgiarginfo::must;
  argsinfo.addarginfo (NULL, arg_ainfo);


  // "ifl" - I'm feeling lucky! (Go directly to a matching document)
  arg_ainfo.shortname = "ifl";
  arg_ainfo.longname = "i'm feeling lucky";
  arg_ainfo.multiplechar = false;
  arg_ainfo.multiplevalue = false;
  arg_ainfo.defaultstatus = cgiarginfo::weak;
  arg_ainfo.argdefault = g_EmptyText;
  arg_ainfo.savedarginfo = cgiarginfo::mustnot;
  argsinfo.addarginfo (NULL, arg_ainfo);

  // "ifln" - I'm feeling lucky number (Go directly to the nth matching document)
  arg_ainfo.shortname = "ifln";
  arg_ainfo.longname = "i'm feeling lucky number";
  arg_ainfo.multiplechar = true;
  arg_ainfo.multiplevalue = false;
  arg_ainfo.defaultstatus = cgiarginfo::weak;
  arg_ainfo.argdefault = "1";
  arg_ainfo.savedarginfo = cgiarginfo::mustnot;
  argsinfo.addarginfo (NULL, arg_ainfo);

  // "srn" - the next search result
  arg_ainfo.shortname = "srn";
  arg_ainfo.longname = "the next search result number";
  arg_ainfo.multiplechar = true;
  arg_ainfo.multiplevalue = false;
  arg_ainfo.defaultstatus = cgiarginfo::weak;
  arg_ainfo.argdefault = "0";
  arg_ainfo.savedarginfo = cgiarginfo::must;
  argsinfo.addarginfo (NULL, arg_ainfo);

  // "srp" - the previous search result
  arg_ainfo.shortname = "srp";
  arg_ainfo.longname = "the previous search result number";
  arg_ainfo.multiplechar = true;
  arg_ainfo.multiplevalue = false;
  arg_ainfo.defaultstatus = cgiarginfo::weak;
  arg_ainfo.argdefault = "0";
  arg_ainfo.savedarginfo = cgiarginfo::must;
  argsinfo.addarginfo (NULL, arg_ainfo);

  // "sf" - Sort field. Set to field to be used for sorting search reult
  // set (only implemented for lucene collections at present).
  arg_ainfo.shortname = "sf";
  arg_ainfo.longname = "sort field";
  arg_ainfo.multiplechar = true;
  arg_ainfo.multiplevalue = false;
  arg_ainfo.defaultstatus = cgiarginfo::weak;
  arg_ainfo.argdefault = g_EmptyText;
  arg_ainfo.savedarginfo = cgiarginfo::must;
  argsinfo.addarginfo (NULL, arg_ainfo);

  // "so" - sort order. 0 = ascending (natural order), 
  //                    1 = descending (reverse order)
  arg_ainfo.shortname = "so";
  arg_ainfo.longname = "sort order";
  arg_ainfo.multiplechar = false;
  arg_ainfo.multiplevalue = false;
  arg_ainfo.defaultstatus = cgiarginfo::weak;
  arg_ainfo.argdefault = "0";
  arg_ainfo.savedarginfo = cgiarginfo::must;
  argsinfo.addarginfo (NULL, arg_ainfo);

  // "fqn" - number of fields in the query form
  arg_ainfo.shortname = "fqn";
  arg_ainfo.longname = "form query num fields";
  arg_ainfo.multiplechar = true;
  arg_ainfo.multiplevalue = false;
  arg_ainfo.defaultstatus = cgiarginfo::weak;
  arg_ainfo.argdefault = "4";
  arg_ainfo.savedarginfo = cgiarginfo::must;
  argsinfo.addarginfo (NULL, arg_ainfo);

  // "fqf" - the list of field names in the form query
  // - a comma separated list
  arg_ainfo.shortname = "fqf";
  arg_ainfo.longname = "form query fields";
  arg_ainfo.multiplechar = true;
  arg_ainfo.multiplevalue = false;
  arg_ainfo.defaultstatus = cgiarginfo::weak;
  arg_ainfo.argdefault = g_EmptyText;
  arg_ainfo.savedarginfo = cgiarginfo::must;
  argsinfo.addarginfo (NULL, arg_ainfo);
  
  // "fqv" - the list of values in the form query
  // - a comma separated list
  arg_ainfo.shortname = "fqv";
  arg_ainfo.longname = "form query values";
  arg_ainfo.multiplechar = true;
  arg_ainfo.multiplevalue = false;
  arg_ainfo.defaultstatus = cgiarginfo::weak;
  arg_ainfo.argdefault = g_EmptyText;
  arg_ainfo.savedarginfo = cgiarginfo::must;
  argsinfo.addarginfo (NULL, arg_ainfo);
  

  // "fqc" - the list of boolean operators in the form query
  // - a comma separated list
  arg_ainfo.shortname = "fqc";
  arg_ainfo.longname = "form query combines";
  arg_ainfo.multiplechar = true;
  arg_ainfo.multiplevalue = false;
  arg_ainfo.defaultstatus = cgiarginfo::weak;
  arg_ainfo.argdefault = g_EmptyText;
  arg_ainfo.savedarginfo = cgiarginfo::must;
  argsinfo.addarginfo (NULL, arg_ainfo);

  // "fqa" - form query advanced - for "run query"
  arg_ainfo.shortname = "fqa";
  arg_ainfo.longname = "form query advanced query";
  arg_ainfo.multiplechar = false;
  arg_ainfo.multiplevalue = false;
  arg_ainfo.defaultstatus = cgiarginfo::weak;
  arg_ainfo.argdefault = "0";
  arg_ainfo.savedarginfo = cgiarginfo::must;
  argsinfo.addarginfo (NULL, arg_ainfo);

  // "fuzziness" controls how closely the search terms must match
  // 100 = exact match, 0 = very inexact match (only implemented for Lucene)
  arg_ainfo.shortname = "fuzziness";
  arg_ainfo.longname = "Lucene fuzziness value";
  arg_ainfo.multiplechar = true;
  arg_ainfo.multiplevalue = false;
  arg_ainfo.defaultstatus = cgiarginfo::weak;
  arg_ainfo.argdefault = g_EmptyText;
  arg_ainfo.savedarginfo = cgiarginfo::must;
  argsinfo.addarginfo (NULL, arg_ainfo);

  // "hd" history display  - search history only displayed when
  // this var set to something other than 0
  // this number of records is displayed
  arg_ainfo.shortname = "hd";
  arg_ainfo.longname = "history display";
  arg_ainfo.multiplechar = true;
  arg_ainfo.multiplevalue = false;
  arg_ainfo.defaultstatus = cgiarginfo::weak;
  arg_ainfo.argdefault = "0";
  arg_ainfo.savedarginfo = cgiarginfo::must;
  argsinfo.addarginfo (NULL, arg_ainfo);
	    
  // "hs"  save - set to 1 in query form, so only save when submit
  // query
  // 0 = no save 1 = save
  arg_ainfo.shortname = "hs";
  arg_ainfo.longname = "history save";
  arg_ainfo.multiplechar = false;
  arg_ainfo.multiplevalue = false;
  arg_ainfo.defaultstatus = cgiarginfo::weak;
  arg_ainfo.argdefault = "0";
  arg_ainfo.savedarginfo = cgiarginfo::mustnot;
  argsinfo.addarginfo (NULL, arg_ainfo);
}

void basequeryaction::configure (const text_t &key, const text_tarray &cfgline) {
  action::configure (key, cfgline);
}

bool basequeryaction::init (ostream &logout) {
  return action::init (logout);
}

bool basequeryaction::check_cgiargs (cgiargsinfoclass &argsinfo, cgiargsclass &args, 
				 recptprotolistclass * /*protos*/, ostream &logout) {


  // check m argument
  int arg_m = args.getintarg("m");
  if (arg_m < -1) {
    logout << "Warning: \"m\" argument less than -1 (" << arg_m << ")\n";
    cgiarginfo *minfo = argsinfo.getarginfo ("m");
    if (minfo != NULL) args["m"] = minfo->argdefault;
  }

  // check o argument
  int arg_o = args.getintarg("o");
  if (arg_o < -1) {
    logout << "Warning: \"o\" argument less than -1 (" << arg_o << ")\n";
    cgiarginfo *oinfo = argsinfo.getarginfo ("o");
    if (oinfo != NULL) args["o"] = oinfo->argdefault;
  }

  // check r argument
  int arg_r = args.getintarg("r");
  if (arg_r < 1) {
    logout << "Warning: \"r\" argument less than 1 (" << arg_r << ")\n";
    cgiarginfo *rinfo = argsinfo.getarginfo ("r");
    if (rinfo != NULL) args["r"] = rinfo->argdefault;
  }

  //check hd argument
  int arg_hd = args.getintarg("hd");
  if (arg_hd <0 ) {
    logout << "Warning: \"hd\" argument less than 0 (" << arg_hd << ")\n";
    cgiarginfo *hdinfo = argsinfo.getarginfo ("hd");
    if (hdinfo != NULL) args["hd"] = hdinfo->argdefault;
  }  
  
  //check hs argument
  int arg_hs = args.getintarg("hs");
  if (arg_hs !=0 && arg_hs !=1) {
    logout << "Warning: \"hs\" argument out of range (" << arg_hs << ")\n";
    cgiarginfo *hsinfo = argsinfo.getarginfo ("hs");
    if (hsinfo != NULL) args["hs"] = hsinfo->argdefault;
  }

  return true;
}

void basequeryaction::get_cgihead_info (cgiargsclass &args, recptprotolistclass * /*protos*/,
				    response_t &response, text_t &response_data, 
				    ostream &/*logout*/) {
  // If this is an "I'm feeling lucky" request, we don't know the target location until later
  if (!args["ifl"].empty()) {
    response = undecided_location;
    return;
  }

  response = content;
  response_data = "text/html";
}




void basequeryaction::define_internal_macros (displayclass &disp, cgiargsclass &args, 
					      recptprotolistclass * protos, 
					      ostream &logout) 
{
  // The following macros are set later (in define_query_macros) as they can't be set until 
  // the query has been done.
  // _quotedquery_   the part of the query string that was quoted for post-processing
  // _freqmsg_      the term frequency string

  // _resultline_   the "x documents matched the query" string

  // _prevfirst_    these are used when setting up the links to previous/next
  // _prevlast_     pages of results (_thisfirst_ and _thislast_ are used to set
  // _nextfirst_    the 'results x-x for query: xxxx' string in the title bar)
  // _nextlast_
  // _thisfirst_
  // _thislast_

  
  define_form_macros(disp, args, protos, logout);
}





// define_query_macros sets the macros that couldn't be set until the
// query had been done. Those macros are 
// _resultline_, _nextfirst_, _nextlast_, _prevfirst_, _prevlast_,
// _thisfirst_, and _thislast_ and _quotedquery_ 
// this has been simplified so it can be used with both search_single_coll
// and search_multiple_coll
void basequeryaction::define_query_macros (cgiargsclass &args, 
					   displayclass &disp,
					   int numdocs, isapprox isApprox) {
  
  // set up _resultline_ macro
  text_t resline;
  int maxdocs = args.getintarg("m");
  if (maxdocs == -1) maxdocs = numdocs;
  else if (numdocs > maxdocs) {
    numdocs = maxdocs;
    isApprox = MoreThan;
  }

  if (isApprox == Approximate) resline = "_textapprox_";
  else if (isApprox == MoreThan) resline = "_textmorethan_";
  
  if (numdocs == 0) resline = "_textnodocs_";
  else if (numdocs == 1) resline += "_text1doc_";
  else resline += text_t(numdocs) + " _textlotsdocs_";
  
  disp.setmacro("resultline", "query", resline);

  int firstdoc = args.getintarg("r");
  int hitsperpage = args.getintarg("o");
  if (hitsperpage == -1) hitsperpage = numdocs;

  // set up _thisfirst_ and _thislast_ macros
  disp.setmacro ("thisfirst", "query", firstdoc);
  int thislast = firstdoc + (hitsperpage - 1);
  if (thislast > numdocs) thislast = numdocs;
  disp.setmacro ("thislast", "query", thislast);

  // set up _prevfirst_ and _prevlast_ macros
  if (firstdoc > 1) {
    disp.setmacro ("prevlast", "query", firstdoc - 1);
    int prevfirst = firstdoc - hitsperpage;
    if (prevfirst < 1) prevfirst = 1;
    disp.setmacro ("prevfirst", "query", prevfirst);
  }

  // set up _nextfirst_ and _nextlast_ macros
  if (thislast < numdocs) {
    disp.setmacro ("nextfirst", "query", thislast + 1);
    int nextlast = thislast + hitsperpage;
    if (nextlast > numdocs) nextlast = numdocs;
    disp.setmacro ("nextlast", "query", nextlast);
  }
}


// sets the selection box macros such as:
//   _hselection_, _jselection_, _nselection_ _gselection_, _fqfselection_ 

void basequeryaction::set_option_macro (const text_t &macroname, 
					text_t current_value, 
					bool display_single, 
					bool add_js_update,
					const FilterOption_t &option, 
					displayclass &disp) 
{
  if (option.validValues.empty()) return;
  if (option.validValues.size() == 1) {
    if (display_single) {
      disp.setmacro (macroname + "selection", displayclass::defaultpackage, "_" + option.defaultValue + "_");
    }
    return;    
  } 
  if (option.validValues.size() < 2) return;
  
  text_t macrovalue = "<select name=\"" + macroname + "\"";
  if (add_js_update) {
    macrovalue += " onChange=\"update"+macroname+"();\"";
  }
  macrovalue += ">\n";
  
  if (current_value.empty()) current_value = option.defaultValue;
  
  text_tarray::const_iterator thisvalue = option.validValues.begin();
  text_tarray::const_iterator endvalue = option.validValues.end();

  while (thisvalue != endvalue) {
    macrovalue += "<option value=\"" + *thisvalue + "\"";
    if (*thisvalue == current_value)
      macrovalue += " selected";
    macrovalue += ">_" + *thisvalue + "_\n";
    ++thisvalue;
  }
  macrovalue += "</select>\n";
  disp.setmacro (macroname + "selection", displayclass::defaultpackage, macrovalue);
}



// sets the selection box macros such as:
//   _sqlfqfselection_ 

void basequeryaction::set_option_macro (const text_t &macroname, 
					text_t current_value, 
					bool display_single, 
					bool add_js_update,
					const FilterOption_t &option_domain, 
					const FilterOption_t &option_range, 
					displayclass &disp) 
{
  // this should probably be moved to sqlqueryaction.cpp // *****

  if (option_domain.validValues.empty()) return;
  if (option_range.validValues.empty()) return;

  if (option_range.validValues.size() == 1) {
    if (display_single) {
      disp.setmacro (macroname + "selection", 
		     displayclass::defaultpackage, "_" + option_range.defaultValue + "_");
    }
    return;    
  } 
  if (option_domain.validValues.size() < 2) return;
  if (option_range.validValues.size() < 2) return;
  
  text_t macrovalue = "<select name=\"" + macroname + "\"";
  if (add_js_update) {
    macrovalue += " onChange=\"update"+macroname+"();\"";
  }
  macrovalue += ">\n";
  
  if (current_value.empty()) current_value = option_domain.defaultValue;

  text_tarray::const_iterator dom_thisvalue = option_domain.validValues.begin();
  text_tarray::const_iterator dom_endvalue = option_domain.validValues.end();
  
  text_tarray::const_iterator ran_thisvalue = option_range.validValues.begin();
  text_tarray::const_iterator ran_endvalue = option_range.validValues.end();


  while ((dom_thisvalue != dom_endvalue) && (ran_thisvalue != ran_endvalue)) {
    if (*ran_thisvalue != "ZZ" && *ran_thisvalue != "TX") {

      text_t option_val = *dom_thisvalue;
      option_val.replace(",","/");
      macrovalue += "<option value=\"" + option_val + "\"";

      if (*dom_thisvalue == current_value)
	macrovalue += " selected";
      macrovalue += ">_" + *ran_thisvalue + "_\n";
    }

    ++dom_thisvalue;
    ++ran_thisvalue;
  }
  macrovalue += "</select>\n";
  disp.setmacro (macroname + "selection", displayclass::defaultpackage, macrovalue);
}




// define_single_query_macros sets the extra macros for search_single_coll 
// that couldn't be set until the query had been done. Those macros are 
// _freqmsg_ and _stopwordsmsg_
void basequeryaction::define_single_query_macros (cgiargsclass &args, 
					      displayclass &disp, 
					      const FilterResponse_t &response) {
  // set up _freqmsg_ and _stopwordsmsg_ macros

  text_t freqmsg = "";
  freqmsg = "_textfreqmsg1_";
  TermInfo_tarray::const_iterator this_term = response.termInfo.begin();
  TermInfo_tarray::const_iterator end_term = response.termInfo.end();
  while (this_term != end_term) {
    freqmsg += (*this_term).term + ": " + (*this_term).freq;
    if ((this_term + 1) != end_term)
      freqmsg += ", ";
    ++this_term;
  }
  disp.setmacro ("freqmsg", "query", freqmsg);

  text_tset::const_iterator this_stopword = response.stopwords.begin();
  text_tset::const_iterator end_stopword = response.stopwords.end();
  if (this_stopword != end_stopword) {
    text_t stopwordsmsg = "_textstopwordsmsg_ ";
    while (this_stopword != end_stopword) {
      if (stopwordsmsg != "_textstopwordsmsg_ ") {
	stopwordsmsg += ", ";
      }
      stopwordsmsg += (*this_stopword);
      ++this_stopword;
    }
    disp.setmacro("stopwordsmsg", "query", stopwordsmsg);
  }
}



void basequeryaction::define_history_macros (displayclass &disp, 
					     cgiargsclass &args, 
					     recptprotolistclass *protos, 
					     ostream &logout) 
{

  // defines the following macros
  // _searchhistorylist_

  text_t historylist;
  int arghd = args.getintarg("hd");
  if (arghd == 0) {
    historylist="";
  }
  else {
    historylist = "<!-- Search History List -->\n";
   
    text_t userid = args["z"];
    text_tarray entries;
    if (get_history_info (userid, entries, dbhome, logout)) {
      int count = 1;
      text_tarray::iterator here = entries.begin();
      text_tarray::iterator end = entries.end();
      int numrecords=(int)entries.size();
      if (numrecords>arghd) { // only display some of them 
	numrecords = arghd;
      }
      historylist += "<form action=\"_gwcgi_\" name=\"HistoryForm\"><table width=\"537\">\n";

      for (int i=0; i<numrecords;++i) {
	text_t query;
	text_t numdocs;
	text_t cgiargs;
	text_t userinfo;
	text_t escquery;
	split_saved_query(entries[i],numdocs,cgiargs);
	parse_saved_args(cgiargs, "q", query); // get query string out
	decode_cgi_arg(query); // un cgisafe it
	escquery = escape_quotes(query); // escape the quotes and newlines
	text_t histvalue = "histvalue";
	histvalue += i;
	disp.setmacro(histvalue, "query", escquery);
	disp.setmacro(histvalue+"Jssafe", "query", encodeForJavascript(escquery));
	format_user_info(cgiargs, userinfo, args, protos, logout);
	
	historylist += "<tr><td align=\"right\">_imagehistbutton_(";
	historylist += i;
	historylist += ")</td>\n";
	historylist += "<td><table border=\"1\" cellspacing=\"0\" ";
	historylist += "cellpadding=\"0\"><tr><td width=\"365\" align=\"left\">"
	  + query 
	  + "</td></tr></table></td><td width=\"110\" align=\"center\"><small>"
	  + numdocs;
	if (numdocs == 1) historylist += " _texthresult_";
	else historylist += " _texthresults_";
	if (!userinfo.empty()) {
	  historylist += "<br>( "+userinfo+" )";
	}
	historylist += "</small></td>\n";
      }
      historylist+="</table></form>\n\n";
     
    } // if get history info 
    else {
      historylist += "_textnohistory_";
    }
    historylist += "<! ---- end of history list ----->\n";
  } // else display list
  disp.setmacro("searchhistorylist", "query", historylist);
  
} // define history macros


bool basequeryaction::search_single_collection (cgiargsclass& args, 
						const text_t& collection,
						recptprotolistclass *protos, 
						browsermapclass* browsers, 
						displayclass& disp, 
						outconvertclass& outconvert, 
						ostream& textout, 
						ostream& logout) 
{
  recptproto *collectproto = protos->getrecptproto (collection, logout);
  if (collectproto == NULL) {
    logout << outconvert << "basequeryaction::search_single_collection: " << collection
	   << " collection has a NULL collectproto\n";

    // Display the "this collection is not installed on this system" page
    disp.setmacro("cvariable", displayclass::defaultpackage, encodeForHTML(collection));
    disp.setmacro("content", "query", "<p>_textbadcollection_<p>");

    textout << outconvert << disp << "_query:header_\n"
	    << "_query:content_\n" << "_query:footer_\n";
    return true;
  }

  comerror_t err;
  ColInfoResponse_t *cinfo = recpt->get_collectinfo_ptr (collectproto, collection, logout);
  
  if (cinfo == NULL) {
    logout << "ERROR (query_action::search_single_collection): get_collectinfo_ptr returned NULL\n";
    return false;
  }
    
  bool segment = cinfo->isSegmented;

  // get the formatted query string
  text_t formattedstring = "";
  get_formatted_query_string(formattedstring, segment, args, disp, logout);

  if (formattedstring.empty()) {
    // no point carrying on if we have no query
    define_history_macros (disp, args, protos, logout);
    textout << outconvert << disp << "_query:header_\n"
	  << "_query:content_";
    textout << outconvert << disp << "_query:footer_";

    return true;
  }
    
  // (.*)queryaction uses "VList" browser to display results,
  // a queries clasification is "Search"
  text_t browsertype = "VList";
  text_t classification = "Search";

  browserclass *bptr = browsers->getbrowser (browsertype);

  // get the formatstring if there is one 
  text_t formatstring;
  if (!get_formatstring (classification, browsertype, 
			 cinfo->format, formatstring)) {
    formatstring = bptr->get_default_formatstring();
  }
  FilterRequest_t request;
  FilterResponse_t response;

  text_t hits_per_page_old = args["o"];
  text_t start_results_from_old = args["r"];  

  // if the "ifl" argument is set to 1, we only want to get one document
  // this may be the first search result (from "I feel lucky") or maybe a 
  // specified search result (from next/prev search result link)
  if (args["ifl"] == "1") {
    args["r"] = args["ifln"]; // the document number we want
    args["o"] = "1";
  }

  bptr->set_filter_options (request, args);
  bptr->load_metadata_defaults (request.fields);

  format_t *formatlistptr = new format_t();
  parse_formatstring (formatstring, formatlistptr, request.fields, request.getParents);
    
  request.filterResultOptions = FROID | FRmetadata | FRtermFreq;
  // note! formattedstring is in unicode! mg and mgpp must convert!
  set_queryfilter_options (request, formattedstring, args);
  
  // do the query
  collectproto->filter (collection, request, response, err, logout);

  // we had an error
  if (err != noError) {
    outconvertclass text_t2ascii;
    logout << text_t2ascii
	   << "basequeryaction::search_single_collections: call to "
	   << query_filter_name() << " failed "
	   << "for " << collection << " collection (" << get_comerror_string (err) << ")\n";
    
    disp.setmacro("resultline", "query", "_textnodocs_");
    if (err == syntaxError) {
      disp.setmacro ("freqmsg", "query", "_textinvalidquery_");
    } else {
      disp.setmacro ("freqmsg", "query", "");
    }
    // restore r and o args in case they have changed
    args["r"] = start_results_from_old;
    args["o"] = hits_per_page_old;
    
    define_history_macros (disp, args, protos, logout);
    
    textout << outconvert << disp << "_query:header_\n"
	    << "_query:content_";
    textout << outconvert << disp << "_query:footer_";
    delete (formatlistptr);
    return true;
  }
  
  // ok, so no error, lets output the results
  
  // Perform the "I'm feeling lucky" trick if the "ifl" argument is set
  if (!args["ifl"].empty()) {
    //Restore the "r" and "o" arg
    args["r"] = start_results_from_old;
    args["o"] = hits_per_page_old;
    
    //Find whether DocumentSearchResultLinks is enabled
    bool show_links = false;
    text_tmap::const_iterator format_here = cinfo->format.begin();
    text_tmap::const_iterator format_end = cinfo->format.end();
    
    while (format_here != format_end) {
      if (((*format_here).first == "DocumentSearchResultLinks") && 
	  ((*format_here).second == "true")){
	show_links = true;
	break;
      }
      ++format_here;
    } 
    
    if (args["ifl"] == 1 || (args["ifl"] == 2 && response.numDocs == 1)) {
      
      // The first search result is the one we want
      if (response.docInfo.begin() != response.docInfo.end()) {
	
	ResultDocInfo_tarray::iterator section = response.docInfo.begin();
	
	// We still need to set "srn" and "srp" values (next and prev search result numbers) if we are showing them
	int srn = 0;
	int srp = 0;
	if (show_links) {
	  int ifln = args["ifln"].getint();
	  srn = ifln + 1;
	  if (srn > response.numDocs) {
	    srn = 0;
	  }
	  srp = ifln - 1;
	  if (srp < 0) {
	    srp = 0;
	  }
	}
	
	textout << outconvert << disp 
		<< "Location: _gwcgi_?e=_compressedoptions_&a=d&c=" 
		<< encodeForURL(collection) << "&cl=search&d=" << (*section).OID
		<< "&srn=" << srn << "&srp=" << srp << "\n\n";
	textout << flush;
	
	return true;
      }
    }
    
    // There weren't enough (or any) matching documents
    // We'll just carry on as if ifl wasn't set. The only catch is that get_cgihead_info won't have
    // done the right thing (because ifl was set), so we need to make sure the output is html
    textout << "Content-type: text/html\n\n";
    
  }

  define_query_macros (args, disp, response.numDocs, response.isApprox);
  define_single_query_macros(args, disp, response);
  // save the query if appropriate
  save_search_history(args, response.numDocs, response.isApprox);
  
  
  // would this result in an error  in err???
  // If Lucene threw a TooManyClauses exception, tell the user about it
  if (args["ct"] == 2 && response.error_message == "TOO_MANY_CLAUSES") {
    cerr << "too many clauses\n";
    disp.setmacro ("freqmsg", "query", "_textlucenetoomanyclauses_");
  }
  

  //Restore the "r" and "o" arg in case they have been changed and we still get here
  args["r"] = start_results_from_old;
  args["o"] = hits_per_page_old;

  define_history_macros (disp, args, protos, logout);

  textout << outconvert << disp << "_query:header_\n"
	  << "_query:content_";
  
  // output the results
  text_t numdocs_t = response.numDocs;  
  args["nmd"] = numdocs_t;
  bool use_table = is_table_content (formatlistptr);
  bptr->output_section_group (response, args, collection, 0, formatlistptr, 
			      use_table, request.fields, request.getParents, 
			      collectproto, disp, outconvert, textout, logout);
  
  
  textout << outconvert << disp << "_query:footer_";
  
  delete (formatlistptr);
  
  return true;
}
 
