/**********************************************************************
 *
 * Jdb2Txt.java -- 
 * A component of the Greenstone digital library software
 * from the New Zealand Digital Library Project at the 
 * University of Waikato, New Zealand.
 *
 * Copyright (C) 1999  The New Zealand Digital Library Project
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 **********************************************************************/

import jdbm.RecordManager;
import jdbm.RecordManagerFactory;
import jdbm.helper.FastIterator;
import jdbm.helper.StringComparator;
import jdbm.htree.HTree;

import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.io.OutputStreamWriter;
import java.io.PrintWriter;
import java.util.Collections;
import java.util.Iterator;
import java.util.Properties;
import java.util.Vector;


public class Jdb2Txt
{
    static String TNAME = "greenstone";

    RecordManager  recman_;
    HTree          hashtable_;
    boolean        sort_;

    static private PrintWriter utf8out = null;

    static
    {
        try {
            OutputStreamWriter osw = new OutputStreamWriter(System.out, "UTF-8");
            utf8out = new PrintWriter(osw, true);
        }
        catch (UnsupportedEncodingException e) {
            System.out.println(e);
        }
    }

    // the default constructor retains default behaviour of not sorting the keys
    public Jdb2Txt(String db_filename)
	throws IOException
    {
	this(db_filename, false);
    }

    public Jdb2Txt(String db_filename, boolean sort)
	throws IOException
    {
	this.sort_ = sort;

	if (db_filename.endsWith(".jdb")) {
	    // remove file extension as JDBM does not expect it
	    db_filename = db_filename.substring(0,db_filename.length()-4);
	}

        // create or open a record manager
        Properties props = new Properties();
        recman_ = RecordManagerFactory.createRecordManager(db_filename, props);

        // create or load table
        long recid = recman_.getNamedObject(TNAME);
        if (recid != 0) {
            hashtable_ = HTree.load(recman_, recid);
        } 
	else {
            System.err.println("Failed to find database table '" + TNAME +"' ...");
	    System.exit(-1);
        }
    }


    public void db2txt()
        throws IOException
    {
	FastIterator   iter;
	String         key;
	String         val;
	
	Vector keylist = sort_ ? new Vector() : null;

        iter = hashtable_.keys();
        key = (String) iter.next();
        while (key != null) {
            val = (String) hashtable_.get(key);

	    if(sort_) {
		keylist.add(key);
	    }
	    else {
		print_entry(key, val);
	    }
            key = (String) iter.next();
        }
        
	if(sort_) {
	    Collections.sort(keylist, new StringComparator());
	    Iterator i = keylist.iterator();
	    while(i.hasNext()) {
		key = (String)i.next();
		val = (String) hashtable_.get(key);
		print_entry(key, val);
	    }
	}

        recman_.close();
    }

    public static void print_entry(String key, String value) {
	utf8out.println("[" + key + "]");
	utf8out.println(value);
	// 70 hypens
	utf8out.println("----------------------------------------------------------------------");
            
    }

    public static void print_usage()
    {
	System.err.println("Usage: java Jdb2Txt [options] database-name");
	System.err.println("options: ");
	System.err.println(" -sort\tsort the keys to the database before output\n");
	System.exit(-1);
    }

   
    public static void main(String[] args)
    {
	if (args.length < 1 || args.length > 2) {
	    print_usage();
	}

	String dbname = args[0];
	boolean sort = false;

	if(args.length == 2) {
	    if(args[0].equals("-sort")) {
		sort = true;
		dbname = args[1];
	    } else {
		System.err.println(args[0] + " is not a valid option\n");
		print_usage();
	    }
	}

        try {

            Jdb2Txt table = new Jdb2Txt(dbname, sort);
            table.db2txt();
        } 
	catch (IOException e) {
            e.printStackTrace();
        }
    }

}
