/**********************************************************************
 *
 * userdb.h -- functions to handle a user database
 * Copyright (C) 1999  DigiLib Systems Limited, New Zealand
 *
 * A component of the Greenstone digital library software
 * from the New Zealand Digital Library Project at the
 * University of Waikato, New Zealand.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *********************************************************************/


#ifndef USERDB_H
#define USERDB_H

#define ERRNO_SUCCEED           0 
#define ERRNO_CONNECTIONFAILED -1
#define ERRNO_USERNOTFOUND     -2
#define ERRNO_EXISTINGUSERNAME -3
#define ERRNO_MISSINGPASSWORD  -4
#define ERRNO_PASSWORDMISMATCH -5
#define ERRNO_DBACTIONFAILED   -6


#include "dbclass.h"

// The user information object and it provides basic functions for the information object, such as clear and copy (=) functions;
class userinfo_t
{
 public:
  text_t username;
  text_t password;
  bool enabled;
  text_t groups; // comma separated list
  text_t comment;

  userinfo_t ();
  virtual ~userinfo_t();
  userinfo_t &operator=(const userinfo_t &x);
  void clear ();
};

typedef vector<userinfo_t> userinfo_tarray; // more space efficient than text_tlist

class userdbclass
{
 protected:
  // The internal storage for the userdbfilename and the userdb object
  dbclass* userdb;
  bool external_db;
  bool activated;
  text_t storeduserdbfilename; 
  
 public:
  userdbclass (const text_t &gsdlhome);
  virtual ~userdbclass();

  //======== the static methods========//
  // the password line in the userinfo_t must be encrypted using
  // this function
  static text_t crypt_text (const text_t &text);
  // username_ok tests to make sure a username is ok. a username
  // must be at least 2 characters long, but no longer than 30
  // characters long. it can contain the characters a-z A-Z 0-9
  // . and _
  static bool username_ok (const text_t &username);
  // password_ok tests to make sure a password is ok. a password
  // must be at least 3 characters long but no longer than 8 characters
  // long. it can contain any character in the range 0x20-0x7e
  static bool password_ok (const text_t &password);
  // removes spaces from user groups
  static text_t format_user_groups(const text_t user_groups);
  
  //======== functions dealing with single user in the user databases ========//
  // returns true on success (in which case userinfo will contain
  // the information for this user)
  int get_user_info (const text_t &username, userinfo_t &userinfo);
  
  // returns true on success
  int set_user_info (const text_t &username, const userinfo_t &userinfo);
  
  // returns true if the user's password is correct.
  int check_passwd (const text_t &username, const text_t &password);
  
  // adds a user from the user database -- forever
  int add_user (const userinfo_t &userinfo);
  
  // edits a user from the user database -- forever
  int edit_user (const userinfo_t &userinfo);
  
  // removes a user from the user database -- forever
  int delete_user (const text_t &username);
  
  //======== functions dealing with multiple users in the user databases ========//
  // gets all the users' information in the database. returns true
  // on success
  int get_all_users(userinfo_tarray &userinfo_array);
  
  // gets a list of all the users in the database. returns true
  // on success
  int get_user_list (text_tarray &userlist);
};

class keydbclass
{
 protected:
  // The internal storage for the userdbfilename and the userdb object
  dbclass* keydb;
  bool external_db;
  bool activated;
   text_t storedkeydbfilename; 
  
 public:
  keydbclass (const text_t &gsdlhome);
  virtual ~keydbclass();
  // functions dealing with databases of temporary keys
  
  // generates a random key for the user, stores it in the database and
  // returns it so that it can be used in page generation
  // returns "" on failure
  text_t generate_key (const text_t &username);
  
  // checks to see if there is a key for this particular user in the
  // database that hasn't decayed. a short decay is used when group
  // is set to administrator
  bool check_key (const userinfo_t &thisuser, const text_t &key, const text_t &group, int keydecay);
  
  // remove_old_keys will remove all keys created more than keydecay ago.
  // use sparingly, it can be quite an expensive function
  void remove_old_keys (int keydecay);
};

#endif
