/**********************************************************************
 *
 * FileSystem.h
 * Copyright (C) 2003  UNESCO
 *
 * A component of the Greenstone digital library software
 * from the New Zealand Digital Library Project at the
 * University of Waikato, New Zealand.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *********************************************************************/

////////////////////////////////////////////////////////////////////////////////
// FileSystem.h -- The generic file pointer routines are a collection of member
//                 functions used to wrap  the underlying file system.

//

  
#ifndef __FILE_SYSTEM__
#define __FILE_SYSTEM__

// Non-platform specific include files
#include <string.h>
#include <stdlib.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <stdio.h>
#include "DLLCode.h"
#include "IsisTypes.h"

namespace FileSystem {
	
// ----------------
// Type definitions 
// ----------------
// typedefs for 64-bit and 32-bit file operations and stream position
#if defined (__64_BIT_FILE_SYSTEM__)
    typedef isis::int64_t FAU;       // 64-bit File Address Unit, physical file address type
    typedef isis::int64_t FAU_t;     // Native 64-bit FAU type
    typedef isis::int64_t StreamPos; // 64-bit stream position
#else // Use the 32-bit version by default
	typedef isis::int32_t FAU;       // 32-bit File Address Unit, physical file address type
	typedef isis::int32_t FAU_t;     // Native 32-bit FAU type
	typedef isis::int32_t StreamPos; // 32-bit stream position
#endif


// Select the underlying file system 
#if defined (__64_BIT_FILE_SYSTEM__)
	#include "FS64.H"              // WIN32 will always use NTFS for large files
	typedef  FPTR64 FPTR;


#elif defined (__WIN32__) && defined (__NTFS__)
	// WIN32 file I/O API
	#include <windows.h>
	#include <io.h>
	#include <string.h>
	#include <stdlib.h>
	#include <fcntl.h>
	#include <sys/types.h>
	#include <sys/stat.h>
	#include <stdio.h>
	#include "gxdtypes.h"
	#include "gxheader.h"

	struct DLL_CODE_API FPTR { // Platform specific file pointer type
		HANDLE fptr;
	};

#else // Use the stdio version by default. Common to all platforms.
	// Non-platform specific stdio include files
	#include <string.h>
	#include <stdlib.h>
	#include <fcntl.h>
	#include <sys/types.h>
	#include <sys/stat.h>
	#include <stdio.h>
	#include "IsisTypes.h"
	//#include "gxheader.h"

	struct DLL_CODE_API FPTR { // File pointer type
		 FILE *fptr;
	};
#endif // File system type



    // NOTE: Any underlying file system used must provide the basic
    // functionallity defined here.
    DLL_CODE_API FPTR            *Create(const char  *fname);
    DLL_CODE_API FPTR            *Open(const char *fname, AccessMode mode);
    DLL_CODE_API int             Close(FPTR *stream);
    DLL_CODE_API int             Flush(FPTR *stream);
    DLL_CODE_API isis::ulong32_t Read(FPTR *stream, void *buf, isis::ulong32_t bytes);
    DLL_CODE_API isis::ulong32_t Write(FPTR *stream, const void *buf, isis::ulong32_t bytes);
    DLL_CODE_API FAU_t           Seek(FPTR *stream, FAU_t, SeekMode mode);
    DLL_CODE_API FAU_t           Tell(FPTR *stream);
    DLL_CODE_API int             Exists(const _TCHAR *fname);
    DLL_CODE_API FAU_t           FileSize(const _TCHAR *fname);
    // DLL_CODE_API int             CanOpenForWriting(const _TCHAR *fname);
    // DLL_CODE_API int             CanOpenReadOnly(const _TCHAR *fname);
    // DLL_CODE_API int             IsReadOnly(const _TCHAR *fname);
}

#endif // __FILE_SYSTEM__
